<?php

if (!function_exists('pinConnectionTimezoneToUtc')) {
    function pinConnectionTimezoneToUtc(PDO $pdo): void {
        $driver = $pdo->getAttribute(PDO::ATTR_DRIVER_NAME);
        if ($driver === 'mysql') {
            $pdo->exec("SET time_zone = '+00:00'");
        } elseif ($driver === 'pgsql') {
            $pdo->exec("SET TIME ZONE 'UTC'");
        }
        // SQLite: no-op
    }
}


function initializeGeneric() {
    date_default_timezone_set('UTC'); // app-wide timezone

    if (!defined('TEST_ENV')) {
        require_once __DIR__ . "/Fcors.php";
        require_once __DIR__ . "/config.php";
        require_once __DIR__ . "/db.php";
    }else{
        require_once __DIR__ . "\..\misc\config.php";
        require_once __DIR__ . "\db.php";
    }

    $dbInfo = getDbInfo();

    if(defined('TEST_ENV') && TEST_ENV === true){
        $inputData['db']['dbLogin'] = $GLOBALS['mockPdo'];
        $inputData['db']['dbApp']   = $GLOBALS['mockPdo'];
        setStripeKey($inputData);
    }else{
        $inputData['db']['dbLogin'] = getDbLogin($dbInfo);
        $inputData['db']['dbApp']   = getDb($dbInfo); 
        if (function_exists('setStripeKey')) {
            setStripeKey($inputData);
        }
    }
    
    $inputData['systemId']      = "1"; 
    $inputData['debugModeFlag']   = "CSD";
    $inputData['requiresAdmin'] = [];

    $testInfo['file'][] = "Generic.php";
    $testInfo['date'][] = date('Y-m-d H:i:s');

    // get session timeout
    $sessionTimeout = ini_get("session.gc_maxlifetime");
    $_SESSION['last_activity'] = time();
    $sessionExpiresAt = $_SESSION['last_activity'] + $sessionTimeout;
    $inputData['sessionTimeOut']  = $sessionExpiresAt;

    // Process GET requests
    if (!empty($_GET)) {
        $inputData = array_merge($inputData, sanitizeInput($_GET));
        $inputData['reqType'] = "GET";
        $testInfo['GET'] = $_GET;

    }

    // Process POST requests
    if (!empty($_POST)) {
        $inputData = array_merge($inputData, sanitizeInput($_POST));
        $inputData['reqType'] = "POST";
        $testInfo['POST'] = $_POST;
    }
    

    // Handle JSON input from other HTTP methods like PUT, PATCH, or POST with JSON
    if (empty($_GET) && empty($_POST)) {
        $rawInput = file_get_contents('php://input');
        $decodedInput = json_decode($rawInput, true);
        if (is_array($decodedInput)) {
            $inputData = array_merge($inputData, sanitizeInput($decodedInput));
            $inputData['reqType'] = "Raw JSON";
        }
    }

    if(isset($inputData['action'])){
        $action = $inputData['action'] ?? null;  // ✅ Prevents notice if `action` is missing
    };

    if(isset($action)){
        if($action === "defaultAction"){
            $data['status'] = 'failed';
            $data['message'] = 'default-action called'; 
            echo json_encode($data);
            exit();
        }
    }

    if(isset($_SESSION['tenantId'])){
        $inputData['tenantId'] = $_SESSION['tenantId'];
    }

    if(isset($_SESSION['roleTenantId'])){
        $inputData['roleTenantId'] = $_SESSION['roleTenantId'];
    }

    if(isset($_SESSION['roleSysAdminTenantId'])){
        $inputData['roleSysAdminTenantId'] = $_SESSION['roleSysAdminTenantId'];
    }

    if(isset($_SESSION['id'])){
        $inputData['userId']    = $_SESSION['id'];
    }

    if(isset($_SESSION['roleValue'])){
        $inputData['roleId'] = $_SESSION['roleValue'];
    }else{
        $inputData['roleId'] = 0;
    }

    if (isset($action)   && $action === "getSessionInfo") {
        $inputData['sessionInfo'] = $_SESSION;
    }

    // Remove this at a later point
    $inputData['sessionInfo'] = $_SESSION;

    $inputData['testInfo']['1'] = $testInfo;

    if (!defined('TEST_ENV') || TEST_ENV === false) { 
        logRequest($inputData);  // ✅ Only log in production, not in tests
    }
    return $inputData;
}


function reinitializeTenantId(array &$inputData): void {
    if(isset($_SESSION['tenantId'])) {
        $inputData['tenantId']      = $_SESSION['tenantId'];
    }
    if(isset($_SESSION['roleTenantId'])) {
        $inputData['roleTenantId']  = $_SESSION['roleTenantId'];
    }
}

// Helper function to sanitize inputs
function sanitizeInput($data) {
    if (is_array($data)) {
        return array_map('sanitizeInput', $data);
    }
    return htmlspecialchars($data);
}
function killSessions() {
    $_SESSION = [];

    // ✅ Ensure session is active before destroying
    if (session_status() === PHP_SESSION_ACTIVE) {
        session_destroy();
        session_write_close(); // Ensure session is fully closed
    }
}
function checkLive(){
    $results['status']  = session_status();
    if(isset($_SESSION['id'])){
        $results['userEnabled'] = true;
        $results['reload']      = null;
        $results['active']      = 1;
        $results['status']      = 'success';
    }else{
        $results['reload']      = 'login';
        $results['active']      = 0;
        $results['status']      = 'failed';
        killSessions();
    }
    return $results;
}
function checkUserStatus($keepLive) {
    if (!isset($keepLive['active']) || $keepLive['active'] != 1) {
        $data['result']['status'] = 'failed';
        $data['result']['message'] = 'User is logged out';
        $data['result']['logout'] = 'true';
    } else{
        $data['result']['logout'] = 'false';
    }
    return $data;
}
function SecurityCheck($inputData) {
    $data = [];
    $action = $inputData['action'] ?? null;
    $protection = $inputData['requiresAdmin'] ?? [];

    $isActionRestricted = is_array($protection) && (
        in_array($action, $protection) ||         // legacy
        isset($protection[$action])               // modern
    );

    if ($isActionRestricted) {
        $inputData['keepLive'] = checkLive();

        if (!isset($_SESSION['id'])) {
            $data = checkUserStatus($inputData['keepLive']);
            $data['result']['status']  = 'failed';
            $data['result']['message'] = 'User is not logged in';
            $data['result']['logout']  = 'true';
            $data['security']          = '0';
            return $data;
        }

        // Role-based access check
        if (isset($protection[$action])) {
            $allowedRoles = $protection[$action];
            $userRole = $_SESSION['roleValue'] ?? null;

            if (!in_array($userRole, $allowedRoles)) {
                $data['result']['status']  = 'failed';
                $data['result']['message'] = 'Access denied: insufficient permissions';
                $data['security']          = '0';
                return $data;
            }

            $nestedTenantId = findNestedValueByKey($inputData, 'tenantId');
            if(isset($nestedTenantId)){
            // Tenant check for roles 4 and above
                if ($userRole >= 4) {
                    // Inject session tenantId into $inputData for matching
                    if (!checkTenantIdMatches($inputData)) {
                        $data['result']['status']  = 'failed';
                        $data['result']['message'] = 'Access denied: tenant mismatch';
                        $data['security']          = '0';
                        return $data;
                    }
                }
            }
        }

        $data['security'] = '1'; // Passed all checks
    } else {
        $data['security'] = '1'; // No restrictions
    }

    return $data;
}
function failed($inputData){
    $data['result']['status']   = 'failed';
    $data['result']['message']  = 'No Action Found';
    $data['result']['action']   = $inputData['action'] ?? 'Set-Action-At-Failed';
    return $data;
}
function logRequest($inputData) {
    if (empty($inputData['action']) || !isset($inputData['db']['dbApp'])) {
        return; // ✅ Skip logging if action is missing or DB is not set
    }

    $pdo = $inputData['db']['dbApp'];

    $sql = "INSERT INTO `request_log` (`action`, `userId`, `ip`, `time`, `url`) 
            VALUES (:action, :userId, :ip, :time, :url)";
    
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':action', $inputData['action']);
    $stmt->bindValue(':userId', $inputData['userId'] ?? null);
    $stmt->bindValue(':ip', $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1');
    $stmt->bindValue(':time', date('Y-m-d H:i:s'));
    $stmt->bindValue(':url', $_SERVER['HTTP_REFERER'] ?? null);

    $stmt->execute();
}
function propagateStatusAndMessage(array $input): array {
    $statusPriority = ['failed' => 1, 'warning' => 2, 'success' => 3, 'info' => 4];
    $finalStatus = $input['status'] ?? null;
    $messageList = [];

    // Recursive collector
    $collect = function ($array) use (&$collect, &$finalStatus, &$messageList, $statusPriority) {
        if (!is_array($array)) return;

        $currentStatus = $array['status'] ?? null;
        $currentMessage = trim($array['message'] ?? '');

        // If current status is failed, collect message if not already in list
        if ($currentStatus === 'failed' && $currentMessage !== '') {
            if (!in_array($currentMessage, $messageList, true)) {
                $messageList[] = $currentMessage;
            }
        }

        // Update top-level status with most critical
        if ($currentStatus !== null) {
            if (!isset($finalStatus) || ($statusPriority[$currentStatus] ?? 999) < ($statusPriority[$finalStatus] ?? 999)) {
                $finalStatus = $currentStatus;
            }
        }

        // Dive into nested arrays
        foreach ($array as $value) {
            if (is_array($value)) {
                $collect($value);
            }
        }
    };

    // Begin recursive walk
    $collect($input);

    // Set final status/message
    if (isset($finalStatus)) $input['status'] = $finalStatus;
    if (!empty($messageList)) $input['message'] = implode(' | ', $messageList);

    // Flatten result array (excluding status/message)
    if (isset($input['result']) && is_array($input['result'])) {
        foreach ($input['result'] as $key => $value) {
            if (!in_array($key, ['status', 'message'])) {
                $input[$key] = $value;
            }
        }
        unset($input['result']);
    }

    return $input;
}
function checkTenantIdMatches($inputData){
    $nestedTenantId = findNestedValueByKey($inputData, 'tenantId');
    if (!isset($nestedTenantId) || !isset($inputData['roleTenantId'])) {
        return false; // Missing tenantId or roleTenantId
    }
    if ($nestedTenantId != $inputData['roleTenantId']) {
        return false; // Tenant IDs do not match
    }
    return true; // Tenant IDs match
}
function findNestedValueByKey(array $array, string $targetKey) {
    foreach ($array as $key => $value) {
        if ($key === $targetKey) {
            return $value;
        }
        if (is_array($value)) {
            $result = findNestedValueByKey($value, $targetKey);
            if ($result !== null) {
                return $result;
            }
        }
    }
    return null;
}
function cleanUpOutputData(array &$data):void {
    foreach ($data as $key => &$value) {
        if ($key === 'secure') {
            unset($data[$key]);
        } elseif (is_array($value)) {
            cleanUpOutputData($value); // recursive call by reference
        }
    }
}
initializeGeneric();
?>