<?php
/**
 * Generator for core/unit-test/phpunit.xml
 * Usage:
 *  php core/unit-test/generate-phpunit.php [--exclude=mod1,mod2]
 * Or set environment variable TEST_EXCLUDE="mod1,mod2"
 */

$scriptDir = __DIR__;
$projectRoot = dirname($scriptDir, 2);

// Parse excludes from env or CLI
$excludes = [];
if (getenv('TEST_EXCLUDE')) {
    $excludes = array_filter(array_map('trim', explode(',', getenv('TEST_EXCLUDE'))));
}

foreach ($argv as $arg) {
    if (str_starts_with($arg, '--exclude=')) {
        $list = substr($arg, 10);
        $parts = array_filter(array_map('trim', explode(',', $list)));
        $excludes = array_merge($excludes, $parts);
    }
}

// Default excluded folders
$excludedFolders = array_merge($excludes, ['vendor', 'tests', 'scripts', 'Default Example', '.git', '.phpunit.cache', 'node_modules']);

$moduleOrder = ['core', 'cms', 'cms-image', 'login', 'tenant', 'food-menu', 'food-booking', 'food-order-public', 'food-order-tenant', 'food-order-super-admin', 'wood-product', 'meta-data'];

$allModules = [];
$dirs = array_filter(glob($projectRoot . '/*'), 'is_dir');

foreach ($dirs as $dir) {
    $moduleName = basename($dir);
    if (in_array($moduleName, $excludedFolders, true)) {
        continue;
    }
    $testsPath = $dir . '/tests';
    if (is_dir($testsPath)) {
        $testFiles = glob($testsPath . '/*.php');
        if (!empty($testFiles)) {
            $allModules[] = [
                'name' => $moduleName,
                'path' => $testsPath,
                'dir' => $dir
            ];
        }
    }
}

// Sort modules by predefined order then append others
$sorted = [];
foreach ($moduleOrder as $m) {
    foreach ($allModules as $mod) {
        if ($mod['name'] === $m) {
            $sorted[] = $mod;
            break;
        }
    }
}
foreach ($allModules as $mod) {
    $found = false;
    foreach ($sorted as $s) {
        if ($s['name'] === $mod['name']) { $found = true; break; }
    }
    if (!$found) $sorted[] = $mod;
}

if (empty($sorted)) {
    echo "⚠️  No test modules found, leaving phpunit.xml unchanged.\n";
    exit(0);
}

// Build xml
$bootstrapPath = "scripts/bootstrap.php";
$xml = '<?xml version="1.0" encoding="UTF-8"?>\n'
    . '<phpunit xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"\n'
    . '         xsi:noNamespaceSchemaLocation="vendor/phpunit/phpunit/phpunit.xsd"\n'
    . '         bootstrap="' . $bootstrapPath . '"\n'
    . '         colors="true"\n'
    . '         failOnRisky="true"\n'
    . '         failOnWarning="false"\n'
    . '         beStrictAboutOutputDuringTests="false"\n'
    . '         beStrictAboutChangesToGlobalState="false"\n'
    . '         beStrictAboutTestsThatDoNotTestAnything="false"\n'
    . '         cacheDirectory="../../.phpunit.cache">\n\n'
    . '    <testsuites>';

foreach ($sorted as $module) {
    $displayName = ucwords(str_replace(['-', '_'], ' ', $module['name'])) . ' Tests';
    // preserve a few custom names
    $map = [
        'core' => 'Core Tests',
        'cms' => 'cms Tests',
        'cms-image' => 'cms Images Tests',
        'login' => 'Login Tests',
        'tenant' => 'Tenant Tests',
        'food-menu' => 'Food Menu Tests',
        'food-booking' => 'Food Booking Tests',
        'food-order-public' => 'Food Order Public Tests',
        'food-order-tenant' => 'Food Order Tenant Tests',
        'food-order-super-admin' => 'Food Order Super Admin Tests',
        'wood-product' => 'Wood Product Tests',
        'meta-data' => 'Meta Data Tests'
    ];
    if (isset($map[$module['name']])) $displayName = $map[$module['name']];

    $relativePath = '../../' . $module['name'] . '/tests';
    $xml .= "\n        <testsuite name=\"{$displayName}\">";
    $xml .= "\n            <directory>{$relativePath}</directory>";
    $xml .= "\n        </testsuite>";
}

$xml .= '\n    </testsuites>\n\n    <source>\n        <include>\n';

// Add source include directories only for modules that exist and have src
foreach ($sorted as $module) {
    $srcDir = $projectRoot . DIRECTORY_SEPARATOR . $module['name'] . DIRECTORY_SEPARATOR . 'src';
    if (is_dir($srcDir)) {
        $xml .= "            <directory>../../{$module['name']}/src</directory>\n";
    }
}

$xml .= '        </include>\n    </source>\n\n</phpunit>';

$outPath = $scriptDir . '/phpunit.xml';
if (file_put_contents($outPath, $xml) === false) {
    echo "❌ Failed to write {$outPath}\n";
    exit(1);
}

echo "✅ Generated phpunit.xml with " . count($sorted) . " test suites\n";
exit(0);

?>
