<?php
/**
 * Orbit Core Unit Testing Framework - Cross-Platform Runner
 * =========================================================
 * 
 * This script provides centralized testing for all modules in the project.
 * It automatically discovers test suites from individual modules and runs them
 * with a shared database schema while preventing function conflicts.
 * 
 * Features:
 * - Auto-generation of phpunit.xml with discovered test suites
 * - Shared SQLite database for all tests with schema loaded once  
 * - Individual test suite execution to prevent PHP function conflicts
 * - Comprehensive logging and reporting
 * - Module dependency ordering based on composer.json
 * - Cross-platform compatibility (Windows, Linux, macOS)
 * 
 * Usage: php core/unit-test/runTest.php
 */

// Start output buffering for clean display
ob_start();

echo "🧪 Orbit Core Unit Test Framework\n";
echo "===================================\n";

// Get project root directory
$projectRoot = dirname(__DIR__, 2);
echo "Project Root: {$projectRoot}\n";

// Change to project root
chdir($projectRoot);

// Check if composer test configuration exists
if (!file_exists('composer.test.json')) {
    echo "❌ composer.test.json not found. Please create this file first.\n";
    exit(1);
}

echo "✅ Found composer.test.json\n";

// Set TEST_ENV for application code
putenv('TEST_ENV=1');
$_ENV['TEST_ENV'] = '1';

// Backup original composer files and switch to test configuration
if (file_exists('composer.json')) {
    copy('composer.json', 'composer.json.bak');
}
if (file_exists('composer.lock')) {
    copy('composer.lock', 'composer.lock.bak');
}
copy('composer.test.json', 'composer.json');

echo "📦 Switched to test composer configuration\n";

// Install test dependencies if needed
$phpunitPath = 'vendor/bin/phpunit';
if (PHP_OS_FAMILY === 'Windows') {
    $phpunitPath = 'vendor\\bin\\phpunit.bat';
}

if (!file_exists($phpunitPath)) {
    echo "📦 Installing test dependencies...\n";
    $output = [];
    $returnCode = 0;
    exec('composer install --no-dev 2>&1', $output, $returnCode);
    
    if ($returnCode !== 0) {
        echo "❌ Failed to install dependencies:\n";
        echo implode("\n", $output) . "\n";
        exit(1);
    }
    echo "✅ Dependencies installed successfully\n";
}

// Scan for test directories in all modules
echo "\n🔍 Scanning for test modules...\n";

$excludedFolders = ['vendor', 'tests', 'scripts', 'Default Example', '.git', '.phpunit.cache', 'node_modules'];
$moduleOrder = ['core', 'cms', 'cms-image', 'login', 'tenant', 'food-menu', 'food-booking', 'food-order-public', 'food-order-tenant', 'food-order-super-admin', 'wood-product', 'meta-data'];

$allModules = [];
$directories = array_filter(glob($projectRoot . '/*'), 'is_dir');

foreach ($directories as $dir) {
    $moduleName = basename($dir);
    
    if (in_array($moduleName, $excludedFolders)) {
        continue;
    }
    
    $testsPath = $dir . '/tests';
    if (is_dir($testsPath)) {
        $testFiles = glob($testsPath . '/*.php');
        if (!empty($testFiles)) {
            // Check if module has its own composer.json for additional module info
            $hasModuleComposer = file_exists($dir . '/composer.json');
            
            $allModules[] = [
                'name' => $moduleName,
                'path' => $testsPath,
                'dir' => $dir,
                'hasComposer' => $hasModuleComposer
            ];
        }
    }
}

// Sort modules by dependency order
$sortedModules = [];
foreach ($moduleOrder as $orderedModule) {
    foreach ($allModules as $module) {
        if ($module['name'] === $orderedModule) {
            $sortedModules[] = $module;
            break;
        }
    }
}

// Add any remaining modules not in the order list
foreach ($allModules as $module) {
    if (!in_array($module['name'], $moduleOrder)) {
        $sortedModules[] = $module;
    }
}

$modules = $sortedModules;

echo "Found " . count($modules) . " test modules:\n";
foreach ($modules as $module) {
    echo "  • {$module['name']}\n";
}

if (empty($modules)) {
    echo "❌ No test modules found. Ensure modules have tests/ directories with PHP test files.\n";
    exit(1);
}

// Generate phpunit.xml configuration using centralized generator
echo "\n📝 Generating phpunit.xml configuration via generator...\n";

$out = [];
$rc = 0;
exec('php core/unit-test/generate-phpunit.php 2>&1', $out, $rc);
if ($rc !== 0) {
    echo "❌ phpunit.xml generator failed:\n" . implode("\n", $out) . "\n";
    exit(1);
}
echo implode("\n", $out) . "\n";
echo "✅ Generated phpunit.xml\n";

// Generate timestamp for logs
$timestamp = date('Y-m-d_H-i-s');
$timestamp2 = date('Y-m-d H:i:s');

// Define log file paths
$reportsDir = "core/unit-test/reports";
$testDoxLog = "{$reportsDir}/testdox/testdox-log_{$timestamp}.txt";
$junitLog = "{$reportsDir}/junit/junit-log_{$timestamp}.xml";
$warningsLog = "{$reportsDir}/warning/warnings-log_{$timestamp}.txt";
$fullLog = "{$reportsDir}/full-output_{$timestamp}.log";

// Ensure reports directories exist
$reportsDirs = [
    "{$reportsDir}/testdox",
    "{$reportsDir}/junit", 
    "{$reportsDir}/warning"
];

foreach ($reportsDirs as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

// Run PHPUnit tests
echo "\n🧪 Running tests...\n";
echo "Test started at {$timestamp2}\n";

// Load phpunit.xml to get test suites
$phpunitXml = simplexml_load_file('core/unit-test/phpunit.xml');
$testsuites = $phpunitXml->testsuites->testsuite;

echo "Running " . count($testsuites) . " test suites individually...\n";

$failedSuites = [];
$totalTests = 0;
$failedTests = 0;

// Build phpunit command based on OS
$phpunitCmd = PHP_OS_FAMILY === 'Windows' ? 'vendor\\bin\\phpunit.bat' : 'vendor/bin/phpunit';

foreach ($testsuites as $testsuite) {
    $suiteName = (string)$testsuite['name'];
    echo "\n🧪 Running: {$suiteName}\n";
    
    // Build command with proper escaping
    $cmd = sprintf(
        '%s --configuration "core/unit-test/phpunit.xml" --testsuite "%s" --testdox --log-junit "%s" --testdox-text "%s" --display-warnings 2> "%s"',
        $phpunitCmd,
        $suiteName,
        $junitLog,
        $testDoxLog,
        $warningsLog
    );
    
    // Execute and capture output
    $output = [];
    $returnCode = 0;
    exec($cmd . ' 2>&1', $output, $returnCode);
    
    // Write output to full log
    file_put_contents($fullLog, implode("\n", $output) . "\n", FILE_APPEND);
    
    // Display output in real-time
    echo implode("\n", $output) . "\n";
    
    if ($returnCode === 0) {
        echo "✅ {$suiteName} completed successfully\n";
    } else {
        echo "❌ {$suiteName} failed with exit code {$returnCode}\n";
        $failedSuites[] = [
            'name' => $suiteName,
            'exitCode' => $returnCode
        ];
    }
}

// Cleanup: remove the temporary SQLite file  
if (file_exists('core/unit-test/scripts/cleanup.php')) {
    include 'core/unit-test/scripts/cleanup.php';
}

// Display summary
echo "\n";
if (!empty($failedSuites)) {
    echo "📊 TEST FAILURE SUMMARY\n";
    echo "========================\n";
    
    foreach ($failedSuites as $failed) {
        echo "❌ {$failed['name']}: Failed with exit code {$failed['exitCode']}\n";
    }
    
    echo "\nFailed Suites: " . count($failedSuites) . " out of " . count($testsuites) . "\n";
} else {
    echo "🎉 ALL TESTS PASSED!\n";
    echo "Successful: " . count($testsuites) . "/" . count($testsuites) . " test suites completed\n";
}

// Display Summary
echo "\nTest completed at " . date('Y-m-d H:i:s') . "\n";
echo "Full Log: {$fullLog}\n";
echo "Warnings Log: {$warningsLog}\n";
echo "TestDox Log: {$testDoxLog}\n";

// Restore original composer files
copy('composer.json', 'composer.test.json');

if (file_exists('composer.json.bak')) {
    copy('composer.json.bak', 'composer.json');
    unlink('composer.json.bak');
}
if (file_exists('composer.lock.bak')) {
    copy('composer.lock.bak', 'composer.lock');
    unlink('composer.lock.bak');
}

echo "✅ Restored original composer configuration\n";

// Clean up environment
unset($_ENV['TEST_ENV']);

// Exit with appropriate code
$exitCode = empty($failedSuites) ? 0 : 1;
exit($exitCode);
?>