<?php
declare(strict_types=1);

// ------------------------------------------------------------
// Load config (keep your current behaviour)
// ------------------------------------------------------------
if (file_exists('.config.php')) {
    include_once('.config.php');
} elseif (file_exists('php/.config.php')) {
    include_once('php/.config.php');
} else {
    http_response_code(500);
    exit("❌ Config file not found in either location");
}

// ------------------------------------------------------------
// Only allow POST
// ------------------------------------------------------------
if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
    http_response_code(405);
    exit("❌ Method not allowed");
}

// ------------------------------------------------------------
// Ensure required config functions exist
// ------------------------------------------------------------
foreach (['unzipCred', 'unzipWindowSeconds', 'unzipAllowedFiles'] as $fn) {
    if (!function_exists($fn)) {
        http_response_code(500);
        exit("❌ Server not configured (missing {$fn})");
    }
}

$secret        = (string) unzipCred();
$windowSeconds = (int) unzipWindowSeconds();
$allowedFiles  = (array) unzipAllowedFiles();

if ($secret === '' || $windowSeconds <= 0 || empty($allowedFiles)) {
    http_response_code(500);
    exit("❌ Server not configured (bad unzip settings)");
}

// ------------------------------------------------------------
// Read inputs
// ------------------------------------------------------------
$fileIn = isset($_POST['file']) ? (string)$_POST['file'] : '';
$fileIn = basename($fileIn); // sanitize
$tsRaw  = (string)($_POST['ts'] ?? '');
$shaExp = strtolower(trim((string)($_POST['sha256'] ?? '')));
$sig    = strtolower(trim((string)($_POST['sig'] ?? '')));

// ------------------------------------------------------------
// Validate base inputs (except file, which may be AUTO)
// ------------------------------------------------------------
if (!ctype_digit($tsRaw)) {
    http_response_code(400);
    exit("❌ Invalid timestamp");
}
if (!preg_match('/^[a-f0-9]{64}$/', $shaExp)) {
    http_response_code(400);
    exit("❌ Invalid sha256");
}
if (!preg_match('/^[a-f0-9]{64}$/', $sig)) {
    http_response_code(400);
    exit("❌ Invalid signature format");
}

// ------------------------------------------------------------
// Resolve file (explicit or auto)
// ------------------------------------------------------------
$file = '';

if ($fileIn !== '' && strtolower($fileIn) !== 'auto') {
    // explicit file requested
    if (!in_array($fileIn, $allowedFiles, true)) {
        http_response_code(400);
        exit("❌ Invalid file");
    }
    $file = $fileIn;
} else {
    // auto-detect: pick first allowed file that exists
    foreach ($allowedFiles as $candidate) {
        if (is_string($candidate) && $candidate !== '' && file_exists($candidate)) {
            $file = $candidate;
            break;
        }
    }
    if ($file === '') {
        http_response_code(404);
        exit("❌ No deployment zip file found");
    }
}

// ------------------------------------------------------------
// Enforce time window
// ------------------------------------------------------------
$ts = (int)$tsRaw;
if (abs(time() - $ts) > $windowSeconds) {
    http_response_code(403);
    exit("❌ Expired request");
}

// ------------------------------------------------------------
// Verify signature (binds to the RESOLVED filename)
// msg = "file|ts|sha256"
// ------------------------------------------------------------
$msg = $file . '|' . $tsRaw . '|' . $shaExp;
$expectedSig = hash_hmac('sha256', $msg, $secret);

if (!hash_equals($expectedSig, $sig)) {
    http_response_code(403);
    exit("❌ Bad signature");
}

// ------------------------------------------------------------
// Verify zip exists + checksum matches
// ------------------------------------------------------------
if (!file_exists($file)) {
    http_response_code(404);
    exit("❌ File not found: $file");
}

$shaActual = hash_file('sha256', $file);
if (!hash_equals($shaExp, $shaActual)) {
    http_response_code(400);
    exit("❌ Checksum mismatch");
}

// ------------------------------------------------------------
// Unzip with zip-slip protection
// ------------------------------------------------------------
$targetDir = '.';

$zip = new ZipArchive();
if ($zip->open($file) !== TRUE) {
    http_response_code(500);
    exit("❌ Failed to open zip");
}

for ($i = 0; $i < $zip->numFiles; $i++) {
    $name = $zip->getNameIndex($i);
    if ($name === false) continue;

    if (str_contains($name, '..') || str_starts_with($name, '/') || preg_match('#^[A-Za-z]:[\\/]{1}#', $name)) {
        $zip->close();
        http_response_code(400);
        exit("❌ Unsafe zip contents");
    }
}

if (!$zip->extractTo($targetDir)) {
    $zip->close();
    http_response_code(500);
    exit("❌ Failed to extract");
}

$zip->close();
@unlink($file);

echo "✅ Successfully unzipped and cleaned up.";
?>
