<?php
    const ERROR_EMAIL_TO    = 'servererrors@prodigydesign.co.uk';
    const FALLBACK_LOG_FILE = __DIR__ . '/error_monitor.log';
    const APP_ENV           = 'DEV'; // 'DEV' | 'UAT' | 'PROD' | ''

    
    error_reporting(E_ALL);
    ini_set('display_errors', 0);
    ini_set('log_errors', 1);
    
    /* ---------- Helpers ---------- */
    function em_now(): string { return gmdate('c'); }
    function em_domain(): string { return $_SERVER['HTTP_HOST'] ?? 'CLI'; }
    function em_url(): string {
        $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $h = $_SERVER['HTTP_HOST'] ?? 'CLI';
        $u = $_SERVER['REQUEST_URI'] ?? '';
        return ($h === 'CLI') ? 'CLI' : "$scheme://$h$u";
    }
    function em_mod(): string {
        $script = $_SERVER['SCRIPT_FILENAME'] ?? $_SERVER['SCRIPT_NAME'] ?? '';
        return $script ? basename(dirname($script)) : 'unknown';
    }
    function em_mask(array $a): array {
        $o = [];
        foreach ($a as $k => $v) {
            $o[$k] = (is_string($k) && preg_match('/(pass|pwd|token|secret|key|authorization|auth)/i', $k))
                ? '***'
                : (is_scalar($v) ? $v : json_encode($v));
        }
        return $o;
    }
    function em_log(string $line): void {
        @file_put_contents(FALLBACK_LOG_FILE, '[' . em_now() . "] $line\n", FILE_APPEND);
    }
    
    /* ---------- Light dedupe (60s) ---------- */
    function em_should_send(string $fingerprint, int $ttl = 60): bool {
        $file = sys_get_temp_dir() . '/em_dupe_' . md5($fingerprint);
        if (file_exists($file) && (time() - filemtime($file) < $ttl)) {
            return false;
        }
        @touch($file);
        return true;
    }
    
    /* ---------- Email sending ---------- */
    function em_send(string $subject, string $html): bool {
        $from = "no-reply@prodigydesign.co.uk";
        $hdrs  = "From: $from\r\n";
        $hdrs .= "Reply-To: $from\r\n";
        $hdrs .= "MIME-Version: 1.0\r\n";
        $hdrs .= "Content-type: text/html; charset=UTF-8\r\n";
    
        // Try with -f first (Linux hosts)
        $okWithF = @mail(ERROR_EMAIL_TO, $subject, $html, $hdrs, "-f$from");
        if ($okWithF) return true;
    
        // Fallback without -f (Windows/shared hosts)
        return @mail(ERROR_EMAIL_TO, $subject, $html, $hdrs);
    }
    
    /* ---------- HTML formatter (Dark Mode friendly) ---------- */
    function em_html(string $level, string $title, string $detail): string {
        $get  = htmlspecialchars(json_encode(em_mask($_GET ?? []), JSON_PRETTY_PRINT));
        $post = htmlspecialchars(json_encode(em_mask($_POST ?? []), JSON_PRETTY_PRINT));
        $url  = htmlspecialchars(em_url());
        $when = em_now();
    
        return <<<HTML
    <!doctype html>
    <html>
    <body style="margin:0;padding:16px;font-family:Segoe UI,Arial,sans-serif;color:#dddddd;background-color:#1e1e1e;">
      <h3 style="margin:0 0 8px;color:#ffffff;">$title</h3>
      <p style="margin:0 0 12px;color:#cccccc;">
        <strong>Level:</strong> $level &nbsp;|&nbsp;
        <strong>URL:</strong> $url &nbsp;|&nbsp;
        <strong>When:</strong> $when
      </p>
      <pre style="padding:10px;border-radius:6px;white-space:pre-wrap;background-color:#2d2d2d;color:#d4d4d4;line-height:1.35;">$detail</pre>
      <h4 style="margin:16px 0 6px;color:#ffffff;">Request</h4>
      <pre style="padding:10px;border-radius:6px;background-color:#2d2d2d;color:#d4d4d4;line-height:1.35;">GET: $get
    POST: $post</pre>
    </body>
    </html>
    HTML;
    }
    
    /* ---------- Notify ---------- */
    function em_notify(string $level, string $file, int $line, string $msg, ?\Throwable $ex = null): void {
        $prefix  = APP_ENV ? '[' . APP_ENV . '] ' : '';
        $subject = sprintf('%s[%s] %s module @ %s (%s:%d)',
            $prefix, strtoupper($level), em_mod(), em_domain(), basename($file), $line);
    
        $detail  = $ex
            ? sprintf("%s\nin %s:%d\n\nStack:\n%s", $ex->getMessage(), $ex->getFile(), $ex->getLine(), $ex->getTraceAsString())
            : sprintf("%s\nin %s:%d", $msg, $file, $line);
    
        $fingerprint = $subject . '|' . substr($detail, 0, 200);
        if (!em_should_send($fingerprint, 60)) {
            em_log('Suppressed duplicate within 60s: ' . $subject);
            return;
        }
    
        $html = em_html($level, $subject, htmlspecialchars($detail));
        $sent = em_send($subject, $html);
        em_log(($sent ? 'Email sent' : 'Email failed') . ": $subject");
    }
    
    /* ---------- Handlers ---------- */
    set_error_handler(function($errno, $errstr, $errfile, $errline) {
        em_notify('error', $errfile, (int)$errline, $errstr, null);
        return false;
    });
    set_exception_handler(function($e) {
        em_notify('exception', $e->getFile(), (int)$e->getLine(), $e->getMessage(), $e);
    });
    register_shutdown_function(function() {
        $e = error_get_last();
        if ($e && in_array($e['type'], [E_ERROR, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR, E_PARSE], true)) {
            em_notify('fatal', $e['file'], (int)$e['line'], $e['message'], null);
        }
    });
?>
