#!/usr/bin/env bash
# Orbit Unit Test Runner - Bash Wrapper
# ====================================
#
# Comprehensive test runner wrapper for Orbit tests.
# Runs unit tests via runTest.php (if present), logs output to reports/ with timestamp.

# Exit on unset vars (but not on command failures; we want to capture exit codes ourselves)
set -u

# Clear console (best effort)
command -v clear >/dev/null 2>&1 && clear || true

# Change to script directory (project root)
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$SCRIPT_DIR" || exit 1

# Simple color helpers (only if stdout is a TTY)
if [[ -t 1 ]]; then
  C_CYAN=$'\033[36m'
  C_YELLOW=$'\033[33m'
  C_GREEN=$'\033[32m'
  C_RED=$'\033[31m'
  C_GRAY=$'\033[90m'
  C_RESET=$'\033[0m'
else
  C_CYAN=""; C_YELLOW=""; C_GREEN=""; C_RED=""; C_GRAY=""; C_RESET=""
fi

echo "${C_CYAN}Working from: $(pwd)${C_RESET}"
echo "${C_YELLOW}===============================================================${C_RESET}"
echo "${C_YELLOW}Orbit Unit Test Suite - Running All Tests${C_RESET}"
echo "${C_YELLOW}===============================================================${C_RESET}"

exitCode=0

# Test Orbit-Shop (runTest.php in current dir)
if [[ -f "runTest.php" ]]; then
  echo
  echo "${C_GREEN}Running Orbit Tests...${C_RESET}"
  echo "${C_GRAY}-------------------------------------${C_RESET}"

  # Ensure reports directory exists
  mkdir -p "reports"

  # Generate timestamp for log file
  orbitTimestamp="$(date +"%Y-%m-%d_%H-%M-%S")"
  orbitLogFile="reports/orbit-output_${orbitTimestamp}.log"

  # Run tests and capture output (stdout+stderr to console AND to file)
  # Note: PIPESTATUS[0] is exit code of `php` even though we pipe into tee
  php "runTest.php" 2>&1 | tee "$orbitLogFile"
  lastExitCode=${PIPESTATUS[0]}

  if [[ $lastExitCode -ne 0 ]]; then
    exitCode=$lastExitCode
    echo "${C_RED}Tests failed with exit code: $lastExitCode${C_RESET}"
    echo "${C_CYAN}Test log saved to: $orbitLogFile${C_RESET}"
  else
    echo "${C_GREEN}Orbit tests completed successfully${C_RESET}"
    echo "${C_CYAN}Test log saved to: $orbitLogFile${C_RESET}"
  fi
else
  echo "${C_YELLOW}Orbit test runner not found${C_RESET}"
fi

echo
echo "${C_YELLOW}===============================================================${C_RESET}"
if [[ $exitCode -eq 0 ]]; then
  echo "${C_GREEN}All tests completed successfully!${C_RESET}"
else
  echo "${C_RED}Some tests failed. Check output above for details.${C_RESET}"
fi
echo "${C_YELLOW}===============================================================${C_RESET}"

exit "$exitCode"
