<?php
// Orbit Unit Test Runner - PHP Wrapper
// ===================================
//
// Cross-platform test runner for Orbit tests.
// - Runs from script directory
// - Ensures reports/ exists
// - Runs runTest.php if present
// - Logs output to reports/ with timestamp
// - Echoes output to console AND writes to log
// - Exits with the same exit code as the test run

declare(strict_types=1);

function isTty(): bool {
    // Best-effort; fine if it returns false in Jenkins
    if (PHP_OS_FAMILY === 'Windows') {
        return function_exists('sapi_windows_vt100_support') ? true : false;
    }
    return function_exists('posix_isatty') ? @posix_isatty(STDOUT) : false;
}

function color(string $text, string $ansiCode): string {
    static $useColor = null;
    if ($useColor === null) $useColor = isTty();
    return $useColor ? "\033[{$ansiCode}m{$text}\033[0m" : $text;
}

function println(string $line = ''): void {
    echo $line . PHP_EOL;
}

function ensureDir(string $path): void {
    if (!is_dir($path)) {
        if (!@mkdir($path, 0777, true) && !is_dir($path)) {
            throw new RuntimeException("Failed to create directory: {$path}");
        }
    }
}

function timestamp(): string {
    return date('Y-m-d_H-i-s');
}

function runPhpAndTee(string $phpFile, string $logFile): int {
    // Build command safely
    $phpBin = PHP_BINARY;
    $cmd = escapeshellarg($phpBin) . ' ' . escapeshellarg($phpFile);

    $descriptors = [
        0 => ["pipe", "r"], // stdin
        1 => ["pipe", "w"], // stdout
        2 => ["pipe", "w"], // stderr
    ];

    $proc = proc_open($cmd, $descriptors, $pipes, getcwd());

    if (!is_resource($proc)) {
        throw new RuntimeException("Failed to start process: {$cmd}");
    }

    fclose($pipes[0]); // no stdin

    $logHandle = fopen($logFile, 'ab');
    if ($logHandle === false) {
        // Still run, but warn
        println(color("Warning: could not open log file for writing: {$logFile}", "33"));
        $logHandle = null;
    }

    // Non-blocking reads so we can interleave stdout/stderr
    stream_set_blocking($pipes[1], false);
    stream_set_blocking($pipes[2], false);

    $stdout = $pipes[1];
    $stderr = $pipes[2];

    while (true) {
        $read = [];
        if (!feof($stdout)) $read[] = $stdout;
        if (!feof($stderr)) $read[] = $stderr;

        if (!$read) break;

        $write = null;
        $except = null;
        // Wait up to 200ms
        @stream_select($read, $write, $except, 0, 200000);

        foreach ($read as $r) {
            $chunk = fread($r, 8192);
            if ($chunk === '' || $chunk === false) continue;

            // Echo to console
            echo $chunk;

            // Write to log
            if ($logHandle) {
                fwrite($logHandle, $chunk);
            }
        }

        $status = proc_get_status($proc);
        if (!$status['running'] && feof($stdout) && feof($stderr)) {
            break;
        }
    }

    fclose($stdout);
    fclose($stderr);
    if ($logHandle) fclose($logHandle);

    $exitCode = proc_close($proc);
    return is_int($exitCode) ? $exitCode : 1;
}

// ------------------------------------------------------------

try {
    // Clear screen (best effort)
    if (PHP_OS_FAMILY !== 'Windows') {
        @system('clear');
    } else {
        @system('cls');
    }

    // Change to script directory (project root)
    $scriptDir = __DIR__;
    chdir($scriptDir);

    println(color("Working from: " . getcwd(), "36"));
    println(color(str_repeat("=", 63), "33"));
    println(color("Orbit Unit Test Suite - Running All Tests", "33"));
    println(color(str_repeat("=", 63), "33"));

    $exitCode = 0;

    if (file_exists("runTest.php")) {
        println();
        println(color("Running Waitron Tests...", "32"));
        println(color("-------------------------------------", "90"));

        ensureDir("reports");
        $logFile = "reports/shop-test-output_" . timestamp() . ".log";

        $lastExit = runPhpAndTee("runTest.php", $logFile);

        if ($lastExit !== 0) {
            $exitCode = $lastExit;
            println(color("Shop tests failed with exit code: {$lastExit}", "31"));
            println(color("Shop test log saved to: {$logFile}", "36"));
        } else {
            println(color("Waitron-Shop tests completed successfully", "32"));
            println(color("Shop test log saved to: {$logFile}", "36"));
        }
    } else {
        println(color("Waitron test runner not found", "33"));
    }

    println();
    println(color(str_repeat("=", 63), "33"));
    if ($exitCode === 0) {
        println(color("All tests completed successfully!", "32"));
    } else {
        println(color("Some tests failed. Check output above for details.", "31"));
    }
    println(color(str_repeat("=", 63), "33"));

    exit($exitCode);

} catch (Throwable $e) {
    // Hard fail if the runner itself breaks
    fwrite(STDERR, "Runner error: " . $e->getMessage() . PHP_EOL);
    exit(1);
}
