<?php
/**
 * Orbit Core Unit Testing Framework - Cross-Platform Runner
 * =========================================================
 * Usage: php core/unit-test/runTest.php
 */

ob_start();

echo "🧪 Orbit Core Unit Test Framework\n";
echo "===================================\n";

$projectRoot = dirname(__DIR__, 2);
echo "Project Root: {$projectRoot}\n";

chdir($projectRoot);

// Track exit code + whether we swapped composer
$exitCode = 1;                 // default fail unless proven pass
$composerSwapped = false;

// DB path must match your bootstrap definition
$dbPath = sys_get_temp_dir() . DIRECTORY_SEPARATOR . 'phpunit_test.sqlite';
$dbSidecars = [$dbPath, $dbPath . '-wal', $dbPath . '-shm'];

try {
    // Check if composer test configuration exists
    if (!file_exists('composer.test.json')) {
        throw new RuntimeException("❌ composer.test.json not found. Please create this file first.");
    }
    echo "✅ Found composer.test.json\n";

    // Set TEST_ENV for application code
    putenv('TEST_ENV=1');
    $_ENV['TEST_ENV'] = '1';

    // Backup original composer files and switch to test configuration
    if (file_exists('composer.json')) {
        copy('composer.json', 'composer.json.bak');
    }
    if (file_exists('composer.lock')) {
        copy('composer.lock', 'composer.lock.bak');
    }
    copy('composer.test.json', 'composer.json');
    $composerSwapped = true;

    echo "📦 Switched to test composer configuration\n";

    // ✅ Reset DB once (bootstrap should only delete if TEST_DB_RESET=1)
    putenv('TEST_DB_RESET=1');
    $_ENV['TEST_DB_RESET'] = '1';

    // Install test dependencies if needed
    $phpunitPath = (PHP_OS_FAMILY === 'Windows') ? 'vendor\\bin\\phpunit.bat' : 'vendor/bin/phpunit';

    if (!file_exists($phpunitPath)) {
        echo "📦 Installing test dependencies...\n";
        $output = [];
        $returnCode = 0;

        // NOTE:
        // If phpunit is in require-dev of composer.test.json, --no-dev will NOT install it.
        // If you hit "phpunit not found", remove --no-dev.
        exec('composer install --no-dev 2>&1', $output, $returnCode);

        if ($returnCode !== 0) {
            throw new RuntimeException("❌ Failed to install dependencies:\n" . implode("\n", $output));
        }
        echo "✅ Dependencies installed successfully\n";
    }

    // Scan for test directories in all modules
    echo "\n🔍 Scanning for test modules...\n";

    $excludedFolders = ['vendor', 'tests', 'scripts', 'Default Example', '.git', '.phpunit.cache', 'node_modules'];
    $moduleOrder = ['core', 'cms', 'cms-image', 'login', 'tenant', 'food-menu', 'food-booking', 'food-order-public', 'food-order-tenant', 'food-order-super-admin', 'wood-product', 'meta-data'];

    $allModules = [];
    $directories = array_filter(glob($projectRoot . '/*'), 'is_dir');

    foreach ($directories as $dir) {
        $moduleName = basename($dir);

        if (in_array($moduleName, $excludedFolders, true)) {
            continue;
        }

        $testsPath = $dir . '/tests';
        if (is_dir($testsPath)) {
            $testFiles = glob($testsPath . '/*.php');
            if (!empty($testFiles)) {
                $hasModuleComposer = file_exists($dir . '/composer.json');

                $allModules[] = [
                    'name' => $moduleName,
                    'path' => $testsPath,
                    'dir'  => $dir,
                    'hasComposer' => $hasModuleComposer
                ];
            }
        }
    }

    // Sort modules by dependency order
    $sortedModules = [];
    foreach ($moduleOrder as $orderedModule) {
        foreach ($allModules as $module) {
            if ($module['name'] === $orderedModule) {
                $sortedModules[] = $module;
                break;
            }
        }
    }
    foreach ($allModules as $module) {
        if (!in_array($module['name'], $moduleOrder, true)) {
            $sortedModules[] = $module;
        }
    }

    $modules = $sortedModules;

    echo "Found " . count($modules) . " test modules:\n";
    foreach ($modules as $module) {
        echo "  • {$module['name']}\n";
    }

    if (empty($modules)) {
        throw new RuntimeException("❌ No test modules found. Ensure modules have tests/ directories with PHP test files.");
    }

    // Generate phpunit.xml configuration using centralized generator
    echo "\n📝 Generating phpunit.xml configuration via generator...\n";
    $out = [];
    $rc = 0;
    exec('php core/unit-test/generate-phpunit.php 2>&1', $out, $rc);

    if ($rc !== 0) {
        throw new RuntimeException("❌ phpunit.xml generator failed:\n" . implode("\n", $out));
    }

    echo implode("\n", $out) . "\n";
    echo "✅ Generated phpunit.xml\n";

    // Generate timestamp for logs
    $timestamp = date('Y-m-d_H-i-s');
    $timestamp2 = date('Y-m-d H:i:s');

    // Define log file paths
    $reportsDir  = "core/unit-test/reports";
    $testDoxLog  = "{$reportsDir}/testdox/testdox-log_{$timestamp}.txt";
    $junitLog    = "{$reportsDir}/junit/junit-log_{$timestamp}.xml";
    $warningsLog = "{$reportsDir}/warning/warnings-log_{$timestamp}.txt";
    $fullLog     = "{$reportsDir}/full-output_{$timestamp}.log";

    // Ensure reports directories exist
    $reportsDirs = [
        "{$reportsDir}/testdox",
        "{$reportsDir}/junit",
        "{$reportsDir}/warning"
    ];
    foreach ($reportsDirs as $dir) {
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
    }

    // Run PHPUnit tests
    echo "\n🧪 Running tests...\n";
    echo "Test started at {$timestamp2}\n";

    $phpunitXml = simplexml_load_file('core/unit-test/phpunit.xml');
    $testsuites = $phpunitXml->testsuites->testsuite;

    echo "Running " . count($testsuites) . " test suites individually...\n";

    $failedSuites = [];
    $phpunitCmd = (PHP_OS_FAMILY === 'Windows') ? 'vendor\\bin\\phpunit.bat' : 'vendor/bin/phpunit';

    $i = 0;
    foreach ($testsuites as $testsuite) {
        $i++;
        $suiteName = (string)$testsuite['name'];
        echo "\n🧪 Running: {$suiteName}\n";

        // ✅ After the first suite starts, disable reset so subsequent suites keep the shared DB
        if ($i === 2) {
            putenv('TEST_DB_RESET=0');
            $_ENV['TEST_DB_RESET'] = '0';
            echo "🔒 DB reset disabled for remaining suites (shared DB preserved)\n";
        }

        $cmd = sprintf(
            '%s --configuration "core/unit-test/phpunit.xml" --testsuite "%s" --testdox --log-junit "%s" --testdox-text "%s" --display-warnings 2> "%s"',
            $phpunitCmd,
            $suiteName,
            $junitLog,
            $testDoxLog,
            $warningsLog
        );

        $output = [];
        $returnCode = 0;
        exec($cmd . ' 2>&1', $output, $returnCode);

        file_put_contents($fullLog, implode("\n", $output) . "\n", FILE_APPEND);
        echo implode("\n", $output) . "\n";

        if ($returnCode === 0) {
            echo "✅ {$suiteName} completed successfully\n";
        } else {
            echo "❌ {$suiteName} failed with exit code {$returnCode}\n";
            $failedSuites[] = ['name' => $suiteName, 'exitCode' => $returnCode];
        }
    }

    // Summary
    echo "\n";
    if (!empty($failedSuites)) {
        echo "📊 TEST FAILURE SUMMARY\n";
        echo "========================\n";
        foreach ($failedSuites as $failed) {
            echo "❌ {$failed['name']}: Failed with exit code {$failed['exitCode']}\n";
        }
        echo "\nFailed Suites: " . count($failedSuites) . " out of " . count($testsuites) . "\n";
        $exitCode = 1;
    } else {
        echo "🎉 ALL TESTS PASSED!\n";
        echo "Successful: " . count($testsuites) . "/" . count($testsuites) . " test suites completed\n";
        $exitCode = 0;
    }

    echo "\nTest completed at " . date('Y-m-d H:i:s') . "\n";
    echo "Full Log: {$fullLog}\n";
    echo "Warnings Log: {$warningsLog}\n";
    echo "TestDox Log: {$testDoxLog}\n";

} catch (Throwable $e) {
    echo "\n❌ Runner error: " . $e->getMessage() . "\n";
    $exitCode = 1;

} finally {
    // ✅ Final DB cleanup (delete once, at the end)
    echo "\n🧹 Final DB cleanup...\n";
    foreach ($dbSidecars as $f) {
        if (file_exists($f)) {
            // Helps on Windows occasionally
            gc_collect_cycles();
            @unlink($f);
        }
    }
    echo "🧹 Deleted (if present): " . implode(', ', $dbSidecars) . "\n";

    // Restore original composer files (always)
    if ($composerSwapped) {
        if (file_exists('composer.json.bak')) {
            copy('composer.json.bak', 'composer.json');
            unlink('composer.json.bak');
        }
        if (file_exists('composer.lock.bak')) {
            copy('composer.lock.bak', 'composer.lock');
            unlink('composer.lock.bak');
        }
        echo "✅ Restored original composer configuration\n";
    }

    // Clean up environment (always)
    unset($_ENV['TEST_ENV']);
    unset($_ENV['TEST_DB_RESET']);
    putenv('TEST_DB_RESET'); // unset in process env
}

exit($exitCode);
