# Orbit Core Unit Testing Framework

This directory contains the centralized unit testing framework for all Orbit modules. The testing infrastructure is designed to be portable and included in every project deployment through the core module.

## Structure

```
core/unit-test/
├── runTest.ps1              # Main test runner script
├── scripts/
│   ├── bootstrap.php        # Test initialization and database setup
│   └── cleanup.php          # Test cleanup and database removal
├── reports/                 # Generated test reports
│   ├── testdox/            # Human-readable test output
│   ├── junit/              # JUnit XML reports
│   ├── warning/            # PHPUnit warnings
│   └── full-output_*.log   # Complete test execution logs
└── README.md               # This file
```

## Features

### 🔧 **Auto-Discovery**
- Automatically scans all modules for `tests/` directories
- Discovers PHP test files and generates phpunit.xml configuration
- Orders test execution based on module dependencies

### 🗄️ **Shared Database**
- Creates a single SQLite database shared across all test suites
- Loads SQL schemas once from numbered SQL files in each module
- Prevents table creation conflicts and improves test performance

### ⚡ **Conflict Prevention**
- Runs each test suite individually to prevent PHP function redeclaration errors
- Maintains isolated execution while sharing the database

### 📊 **Comprehensive Logging**
- TestDox output for human-readable results
- JUnit XML for CI/CD integration
- Full execution logs with timestamps
- Warning capture for debugging

## Usage

### From Project Root
```powershell
.\core\unit-test\runTest.ps1
```

### From Any Directory
```powershell
cd path\to\project
.\core\unit-test\runTest.ps1
```

## Requirements

- **PHP 8.2+** with SQLite extension
- **Composer** with PHPUnit installed via `composer.test.json`
- **PowerShell 5.1+** (Windows) or PowerShell Core (cross-platform)

## Module Requirements

For a module to be included in testing:

1. **Test Directory**: Module must have a `tests/` directory
2. **PHP Test Files**: Directory must contain `.php` test files
3. **SQL Schema** (optional): Numbered SQL files like `01-module.test.sql`
4. **Composer File** (optional): `composer.json` for dependency ordering

### Example Module Structure
```
my-module/
├── composer.json           # Optional: for dependency ordering
├── src/                   # Module source code
├── tests/                 # Required: test directory
│   ├── 05-module.test.sql # Optional: numbered schema
│   └── MyModuleTest.php   # Required: test files
└── ...
```

## SQL Schema Loading

The framework automatically loads SQL schemas from each module in numerical order:

- `00-core.test.sql` (loaded first)
- `01-cms.test.sql` 
- `02-login.test.sql`
- `03-tenant.test.sql`
- ...and so on

Schemas are loaded only once at the beginning of the test run and shared across all test suites.

## Configuration

The framework automatically generates `phpunit.xml` with:

- Bootstrap path: `core/unit-test/scripts/bootstrap.php`
- Test suites for each discovered module
- Source code coverage paths
- Proper UTF-8 encoding and error handling

## Integration

### CI/CD Integration
The framework generates JUnit XML reports in `core/unit-test/reports/junit/` for integration with CI/CD systems.

### IDE Integration
The generated `phpunit.xml` can be used by IDEs like PhpStorm or VS Code for running tests directly from the editor.

## Benefits

### 🎯 **Centralized**
- All testing logic in one place within the core module
- Consistent testing environment across all projects

### 📦 **Portable**
- Travels with the core module to every project
- No need to recreate testing infrastructure for each project

### 🔄 **Maintainable**
- Single location for testing framework updates
- Automatic discovery reduces manual configuration

### ⚡ **Efficient**
- Shared database improves performance
- Individual suite execution prevents conflicts
- One-time schema loading saves time

## Troubleshooting

### Common Issues

1. **"PHPUnit not found"**
   - Ensure `composer.test.json` exists and has PHPUnit dependency
   - Run the script from project root

2. **"No test modules found"**  
   - Ensure modules have `tests/` directories with `.php` files
   - Check that module directories are not in the exclusion list

3. **"Function redeclaration errors"**
   - Framework runs suites individually to prevent this
   - If still occurring, check for global function definitions in test files

4. **SQL schema errors**
   - Ensure SQL files have proper numerical prefixes (e.g., `01-`, `02-`)
   - Check SQL syntax in individual schema files
   - Review bootstrap.php for schema loading logic

### Getting Help

- Check the full execution log in `core/unit-test/reports/full-output_*.log`
- Review warnings in `core/unit-test/reports/warning/warnings-log_*.txt`
- Enable verbose output by modifying the `runTest.ps1` script

---

**Part of the Orbit Framework** | Generated test infrastructure for modular PHP applications