<?php
use PHPUnit\Framework\TestCase;

require_once __DIR__ . '/../../core/unit-test/scripts/bootstrap.php'; // ✅ Use shared mock DB
require_once __DIR__ . '/../../core/src/fcors.php'; 
require_once __DIR__ . '/../../core/src/db.php';
require_once __DIR__ . '/../../core/src/generic.php';
echo "\n>>>>>>>>>>>>>>>>>>>>>>>>>====<<<<<<<<<<<<<<<<<<<<<<<<<\n";
echo "\n\nT02_RequestHandlerTest.php\n";

class T02_RequestHandlerTest extends TestCase {
    private $mockPdo;
    private $inputData;

    protected function setUp(): void {
        // ✅ Use the shared mock database connection
        $this->mockPdo = $GLOBALS['mockPdo'];

        $_SESSION = [];
        $_SERVER['REMOTE_ADDR'] = '127.0.0.1';
        $_SERVER['HTTP_REFERER'] = 'http://localhost/test';

        global $action;
        $action = 'testAction'; 

        // ✅ Use global `$GLOBALS['inputData']` from bootstrap
        $this->inputData = $GLOBALS['inputData'];
    }

    public function test001DatabaseConnection(): void {
        $dbInfo = [
            'host' => 'localhost',
            'sche' => 'test_db',
            'user' => 'test_user',
            'pass' => 'test_pass'
        ];
        $db = getDb($dbInfo);
        $this->assertInstanceOf(PDO::class, $db);
    }
    
    public function test002SessionTimeout(): void {
        $sessionTimeout = (int) ini_get("session.gc_maxlifetime");
        $_SESSION['last_activity'] = time();
        $sessionExpiresAt = $_SESSION['last_activity'] + $sessionTimeout;
        
        $this->assertIsInt($sessionTimeout);
        $this->assertGreaterThan(0, $sessionTimeout);
        $this->assertEquals($_SESSION['last_activity'] + $sessionTimeout, $sessionExpiresAt);
    }
    
    public function test003SanitizeInput(): void {
        $dirtyData = ['name' => '<b>John</b>'];
        $cleanData = sanitizeInput($dirtyData);
        
        $this->assertEquals(htmlspecialchars('<b>John</b>'), $cleanData['name']);
    }
    
    public function test004JsonInputHandling(): void {
        $json = json_encode(['key' => 'value', 'number' => 123]);
        $decodedInput = json_decode($json, true); // Directly decode instead of using php://input
        
        $this->assertIsArray($decodedInput);
        $this->assertArrayHasKey('key', $decodedInput);
        $this->assertEquals('value', $decodedInput['key']);
        $this->assertEquals(123, $decodedInput['number']);
    }
    
    public function test005RequestTypeDetection(): void {
        $_GET['param'] = 'test';
        $_POST = [];
        
        $this->assertEquals('GET', 'GET');
        
        $_GET = [];
        $_POST['param'] = 'test';
        
        $this->assertEquals('POST', 'POST');
    }
    
    public function test006SessionTimeoutEdgeCase(): void {
        if (session_status() !== PHP_SESSION_ACTIVE && headers_sent() === false) {
            ini_set("session.gc_maxlifetime", 0);
        }
        $_SESSION['last_activity'] = time();
        $sessionExpiresAt = $_SESSION['last_activity'];
        
        $this->assertEquals($_SESSION['last_activity'], $sessionExpiresAt);
    }    

    public function test007LogRequest(): void {
        $mockDb = $this->mockPdo;
        
        // Mock request data
        $requestData = [
            'action' => 'POST',
            'userId' => 1,
            'ip' => '127.0.0.1',
            'time' => time(),
            'url' => '/test'
        ];
        
        // Insert mock log entry
        $stmt = $mockDb->prepare("INSERT INTO request_log (action, userId, ip, time, url) VALUES (:action, :userId, :ip, :time, :url)");
        $stmt->execute($requestData);
        
        // Verify the log was inserted
        $stmt = $mockDb->prepare("SELECT * FROM request_log WHERE url = :url");
        $stmt->execute(['url' => '/test']);
        $log = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $this->assertNotEmpty($log);
        $this->assertEquals('POST', $log['action']);
        $this->assertEquals('/test', $log['url']);
        $this->assertEquals('127.0.0.1', $log['ip']);
    }

    public function test008KillSessions(): void {
        if (session_status() !== PHP_SESSION_ACTIVE) {
            session_start();
        }
    
        $_SESSION['user'] = 'testUser';
        $_SESSION['role'] = 'admin';
    
        $this->assertArrayHasKey('user', $_SESSION);
        $this->assertArrayHasKey('role', $_SESSION);
    
        killSessions();
    
        // ✅ Ensure session is active before regenerating the ID
        if (session_status() === PHP_SESSION_ACTIVE) {
            session_regenerate_id(true);
        }
    
        $this->assertEmpty($_SESSION);
    }
    
    public function test009CheckLive(): void {
        if (session_status() !== PHP_SESSION_ACTIVE) {
            session_start();
        }
    
        $_SESSION['id'] = 12345;
        $_SESSION['last_activity'] = time();
        $sessionTimeout = (int) ini_get("session.gc_maxlifetime");
    
        $expectedActiveResponse = ['userEnabled' => true, 'status' => 'success', 'reload' => null, 'active' => 1];
        $this->assertEquals($expectedActiveResponse, checkLive());
    
        $_SESSION['last_activity'] = time() - ($sessionTimeout + 10);
        unset($_SESSION['id']); 
    
        $expectedInactiveResponse = ['status' => 'failed', 'reload' => 'login', 'active' => 0];
        $this->assertEquals($expectedInactiveResponse, checkLive());
    }

    public function test010CheckUserStatus(): void {
        // Simulate user session with an inactive status
        $keepLive = ['active' => 0];
        $expectedResponse = [
            'result' => [
                'status' => 'failed',
                'message' => 'User is logged out',
                'logout' => 'true'
            ]
        ];
        $this->assertEquals($expectedResponse, checkUserStatus($keepLive));

        // Simulate user session with an active status
        $keepLive = ['active' => 1];
        $expectedResponse = [
            'result' => [
                'logout' => 'false'
            ]
        ];
        $this->assertEquals($expectedResponse, checkUserStatus($keepLive));
    }

    public function test011NoActionCall(): void{
        // ✅ Call the function
        $inputData = ['action' => ''];
        $response = failed($inputData);
    
        // ✅ Ensure response is an array
        $this->assertIsArray($response, "Response should be an array.");
        $this->assertArrayHasKey('result', $response, "Response should contain 'result' key.");
        $this->assertIsArray($response['result'], "Result should be an array.");
    
        // ✅ Exact value matches
        $this->assertArrayHasKey('status', $response['result'], "Response should contain 'status' key.");
        $this->assertEquals('failed', $response['result']['status'], "Status should be 'failed'.");
    
        $this->assertArrayHasKey('message', $response['result'], "Response should contain 'message' key.");
        $this->assertEquals('No Action Found', $response['result']['message'], "Message should match expected text.");
    
        // ✅ Check only the existence of 'action' key but ignore its value
        $this->assertArrayHasKey('action', $response['result'], "Response should contain 'action' key.");
    }
    
    public function test012SecurityCheck(): void {
        // Ensure session is started before modifying session data
        if (session_status() === PHP_SESSION_ACTIVE) {
            session_destroy();
        }
        if (session_status() !== PHP_SESSION_ACTIVE) {
            session_start();
        }
        
        // Simulate a non-admin action
        $inputData = ['action' => 'testAction', 'requiresAdmin' => ['adminAction']];
        $expectedResponse = ['security' => '1'];
        $this->assertEquals($expectedResponse, SecurityCheck($inputData));

        // Simulate an admin action with no session ID
        session_destroy(); // Ensure session is destroyed before checking
        session_start(); // Restart session for test integrity
        $inputData = ['action' => 'adminAction', 'requiresAdmin' => ['adminAction']];
        $expectedResponse = [
            'result' => [
                'status' => 'failed',
                'message' => 'User is not logged in',
                'logout' => 'true'
            ],
            'security' => '0'
        ];
        $this->assertEquals($expectedResponse, SecurityCheck($inputData));

        // Simulate an admin action with an active session
        $_SESSION['id'] = 12345;
        $inputData = ['action' => 'adminAction', 'requiresAdmin' => ['adminAction']];
        $expectedResponse = ['security' => '1'];
        $this->assertEquals($expectedResponse, SecurityCheck($inputData));
        
        // Test with an empty requiresAdmin array
        $inputData = ['action' => 'adminAction', 'requiresAdmin' => []];
        $expectedResponse = ['security' => '1'];
        $this->assertEquals($expectedResponse, SecurityCheck($inputData));
        
        // Test with missing 'action' key
        $inputData = ['requiresAdmin' => ['adminAction']];
        $inputData['action'] = $inputData['action'] ?? '';
        $expectedResponse = ['security' => '1'];
        $this->assertEquals($expectedResponse, SecurityCheck($inputData));
        
        // Test with missing 'requiresAdmin' key
        $inputData = ['action' => 'adminAction'];
        $inputData['requiresAdmin'] = $inputData['requiresAdmin'] ?? [];
        $expectedResponse = ['security' => '1'];
        $this->assertEquals($expectedResponse, SecurityCheck($inputData));
        
        // Test with checkLive() returning unexpected data
        $inputData = ['action' => 'adminAction', 'requiresAdmin' => ['adminAction']];
        $mockCheckLiveResponse = ['unexpectedKey' => 'value'];
        $this->assertIsArray($mockCheckLiveResponse);
        
        // Test with completely empty inputData
        $inputData = [];
        $inputData['action'] = $inputData['action'] ?? '';
        $inputData['requiresAdmin'] = $inputData['requiresAdmin'] ?? [];
        $expectedResponse = ['security' => '1'];
        $this->assertEquals($expectedResponse, SecurityCheck($inputData));
        
        // Test with null values in inputData
        $inputData = ['action' => null, 'requiresAdmin' => null];
        $inputData['action'] = $inputData['action'] ?? '';
        $inputData['requiresAdmin'] = $inputData['requiresAdmin'] ?? [];
        $expectedResponse = ['security' => '1'];
        $this->assertEquals($expectedResponse, SecurityCheck($inputData));
    }
    // Some Test for the function removeResultFromArrays in generic.php
    public function test013RemoveResultFromArrays(): void {
        $inputData = [
            'result' => [
                'status' => 'success',
                'message' => 'Test message',
                'data' => [
                    'key1' => 'value1',
                    'key2' => 'value2'
                ]
            ]
        ];
        $expectedOutput = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2'
            ],
            'status' => 'success'
        ];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }

    public function test014RemoveResultFromArraysEmpty(): void {
        $inputData = [];
        $expectedOutput = [];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }

    public function test015RemoveResultFromArraysNoResult(): void {
        $inputData = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2'
            ]
        ];
        $expectedOutput = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2'
            ]
        ];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }
    
    public function test016RemoveResultFromArraysNested(): void {
        $inputData = [
            'result' => [
                'status' => 'success',
                'message' => 'Test message',
                'data' => [
                    'key1' => 'value1',
                    'key2' => 'value2',
                    'nested' => [
                        'nestedKey1' => 'nestedValue1',
                        'nestedKey2' => 'nestedValue2'
                    ]
                ]
            ]
        ];
        $expectedOutput = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2',
                'nested' => [
                    'nestedKey1' => 'nestedValue1',
                    'nestedKey2' => 'nestedValue2'
                ]
            ],
            'status' => 'success',
        ];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }

    // test if failed outside of nest
    public function test017RemoveResultFailedOutsideOfNest(): void {
        $inputData = [
            'result' => [
                'status' => 'failed',
                'message' => 'Test message',
                'data' => [
                    'key1' => 'value1',
                    'key2' => 'value2',
                    'status' => 'success'
                ]
            ]
        ];
        $expectedOutput = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2',
                'status' => 'success'
            ],
            'status' => 'failed',
            'message' => 'Test message'
        ];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }

    // test if failed inside of nest
    public function test018RemoveResultFailedInsideOfNest(): void {
        $inputData = [
            'result' => [
                'status' => 'success',
                'data' => [
                    'key1' => 'value1',
                    'key2' => 'value2',
                    'status' => 'failed',
                    'message' => 'Test message'
                ]
            ]
        ];
        $expectedOutput = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2',
                'status' => 'failed',
                'message' => 'Test message'
            ],
            'status' => 'failed',
            'message' => 'Test message'
        ];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }

    // test if outside of nest sucsess and inside of nest failed
    public function test019RemoveResultSuccessOutsideOfNestFailedInsideOfNest(): void {
        $inputData = [
            'status' => 'failed',
            'message' => 'Test message 1',
            'result' => [
                'status' => 'success',
                'message' => 'Test message 2',
                'data' => [
                    'key1' => 'value1',
                    'key2' => 'value2',
                    'status' => 'failed',
                    'message' => 'Nested message 3'
                ]
            ]
        ];
        $expectedOutput = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2',
                'status' => 'failed',
                'message' => 'Nested message 3'
            ],
            'status' => 'failed',
            'message' => 'Test message 1 | Nested message 3'
        ];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }

    // test if outside of nest failed and inside of nest sucsess
    public function test020RemoveResultFailedOutsideOfNestSuccessInsideOfNest(): void {
        $inputData = [
            'status' => 'success',
            'message' => 'Test message 1',
            'result' => [
                'status' => 'failed',
                'message' => 'Test message 2',
                'data' => [
                    'key1' => 'value1',
                    'key2' => 'value2',
                    'status' => 'success',
                    'message' => 'Nested message 3'
                ]
            ]
        ];
        $expectedOutput = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2',
                'status' => 'success',
                'message' => 'Nested message 3'
            ],
            'status' => 'failed',
            'message' => 'Test message 2'
        ];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }

    // test if sucsess outside of nest and inside of nest 
    public function test021RemoveResultSuccessOutsideOfNestSuccessInsideOfNest(): void {
        $inputData = [
            'status' => 'success',
            'message' => 'Test message 1',
            'result' => [
                'status' => 'success',
                'message' => 'Test message 2',
                'data' => [
                    'key1' => 'value1',
                    'key2' => 'value2',
                    'status' => 'success',
                    'message' => 'Nested message 3'
                ]
            ]
        ];
        $expectedOutput = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2',
                'status' => 'success',
                'message' => 'Nested message 3'
            ],
            'status' => 'success',
            'message' => 'Test message 1'
        ];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }

    // test if failed outside of nest and inside of nest 
    public function test022RemoveResultFailedOutsideOfNestFailedInsideOfNest(): void {
        $inputData = [
            'status' => 'failed',
            'message' => 'Test message 1',
            'result' => [
                'status' => 'failed',
                'message' => 'Test message 2',
                'data' => [
                    'key1' => 'value1',
                    'key2' => 'value2',
                    'status' => 'failed',
                    'message' => 'Nested message 3'
                ]
            ]
        ];
        $expectedOutput = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2',
                'status' => 'failed',
                'message' => 'Nested message 3'
            ],
            'status' => 'failed',
            'message' => 'Test message 1 | Test message 2 | Nested message 3'
        ];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }

    public function test023RemoveResultNoMessageOutsideOfNestFailedInsideOfNest(): void {
        $inputData = [
            'status' => 'success',
            'result' => [
                'status' => 'failed',
                'message' => 'Test message 2',
                'data' => [
                    'key1' => 'value1',
                    'key2' => 'value2',
                    'status' => 'failed',
                    'message' => 'Nested message 3'
                ]
            ]
        ];
        $expectedOutput = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2',
                'status' => 'failed',
                'message' => 'Nested message 3'
            ],
            'status' => 'failed',
            'message' => 'Test message 2 | Nested message 3'
        ];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }

    // duplicate message for failed test
    public function test024RemoveResultDuplicateMessageFailedOutsideOfNestFailedInsideOfNest(): void {
        $inputData = [
            'status' => 'failed',
            'message' => 'Test message 1',
            'result' => [
                'status' => 'failed',
                'message' => 'Test message 1',
                'data' => [
                    'key1' => 'value1',
                    'key2' => 'value2',
                    'status' => 'failed',
                    'message' => 'Nested message 3'
                ]
            ]
        ];
        $expectedOutput = [
            'data' => [
                'key1' => 'value1',
                'key2' => 'value2',
                'status' => 'failed',
                'message' => 'Nested message 3'
            ],
            'status' => 'failed',
            'message' => 'Test message 1 | Nested message 3'
        ];
        $this->assertEquals($expectedOutput, propagateStatusAndMessage($inputData));
    }

    public function test025RoleTestFailed(): void{
        session_start();
        $inputData = [
            'action' => 'adminAction',
            'requiresAdmin' => ['adminAction']
        ];
    
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('0', $res['security']);
        $this->assertEquals('failed', $res['result']['status']);
        $this->assertEquals('User is not logged in', $res['result']['message']);
        $this->assertEquals('true', $res['result']['logout']);
        // Simulate an admin action with no session ID
    }

    public function test026RolePlatform(): void{
        session_start();
        $_SESSION['id'] = 1;
        unset($_SESSION['roleValue']);
        $inputData = [
            'action' => 'adminAction',
            'requiresAdmin' => ['adminAction' => [1,2,3]]
        ];

        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('0', $res['security']);
        $this->assertEquals('failed', $res['result']['status']);
        $this->assertEquals('Access denied: insufficient permissions', $res['result']['message']);

        $_SESSION['roleValue'] = '1';
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('1', $res['security']);

        $_SESSION['roleValue'] = '2';
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('1', $res['security']);

        $_SESSION['roleValue'] = '3';
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('1', $res['security']);

        $_SESSION['roleValue'] = '4';
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('0', $res['security']);
        $this->assertEquals('failed', $res['result']['status']);
        $this->assertEquals('Access denied: insufficient permissions', $res['result']['message']);

        session_destroy();
    }

    public function test027RoleTenant(): void{
        session_start();
        $_SESSION['id'] = 2;
        unset($_SESSION['roleValue']);

        $inputData = [
            'action' => 'adminTenant',
            'requiresAdmin' => ['adminTenant' => [4,5,6,7]]
        ];

        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('0', $res['security']);
        $this->assertEquals('failed', $res['result']['status']);
        $this->assertEquals('Access denied: insufficient permissions', $res['result']['message']);

        $_SESSION['roleValue'] = '4';
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('1', $res['security']);

        $_SESSION['roleValue'] = '5';
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('1', $res['security']);

        $_SESSION['roleValue'] = '6';
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('1', $res['security']);

        $_SESSION['roleValue'] = '7';
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('1', $res['security']);

        $_SESSION['roleValue'] = '1';
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('0', $res['security']);
        $this->assertEquals('failed', $res['result']['status']);
        $this->assertEquals('Access denied: insufficient permissions', $res['result']['message']);

        session_destroy();
    }

    public function test028RolePublic(): void{
        session_start();
        $_SESSION['id'] = 3;
        unset($_SESSION['roleValue']);

        $inputData = [
            'action' => 'viewPublic',
            'requiresAdmin' => ['viewPublic' => [8]]
        ];

        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('0', $res['security']);
        $this->assertEquals('failed', $res['result']['status']);
        $this->assertEquals('Access denied: insufficient permissions', $res['result']['message']);

        $_SESSION['roleValue'] = '8';
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('1', $res['security']);

        $_SESSION['roleValue'] = '1';
        $res = SecurityCheck($inputData);
        //echo(print_r($res,true));
        $this->assertEquals('0', $res['security']);
        $this->assertEquals('failed', $res['result']['status']);
        $this->assertEquals('Access denied: insufficient permissions', $res['result']['message']);

        session_destroy();
    }

}
?>