-- Events Table
CREATE TABLE event_details (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    small_summary TEXT,
    rich_explanation TEXT,
    location VARCHAR(255),
    latitude DECIMAL(9,6) NOT NULL,
    longitude DECIMAL(9,6) NOT NULL,
    start_datetime DATETIME NOT NULL,
    end_datetime DATETIME NOT NULL,
    owner_id INT NOT NULL,
    is_enabled BOOLEAN DEFAULT TRUE,
    formatted_address VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_modified TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (owner_id) REFERENCES user_login(pk_user_id)
) ENGINE=InnoDB;


-- Event Co-Owners Table
CREATE TABLE event_co_owners (
    event_id INT,
    user_id INT,
    PRIMARY KEY (event_id, user_id),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_modified TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (event_id) REFERENCES event_details(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES user_login(pk_user_id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Event Subscribers Table
CREATE TABLE event_subscriptions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    event_id INT NOT NULL,
    user_id INT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    has_uploaded_file BOOLEAN DEFAULT FALSE,
    last_modified TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (event_id) REFERENCES event_details(id) ON DELETE CASCADE
) ENGINE=InnoDB;


-- Event Files Table
-- Event Files Table
CREATE TABLE event_files (
    id INT AUTO_INCREMENT PRIMARY KEY,
    event_id INT NOT NULL,
    uploader_id INT NOT NULL,
    subscriber_id INT,
    file_name VARCHAR(255) NOT NULL,
    file_url VARCHAR(255) NOT NULL,
    is_public BOOLEAN DEFAULT FALSE,
    file_type ENUM('Other', 'Cover Image', 'Event Documentation', 'List Upload') NOT NULL DEFAULT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_modified TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (event_id) REFERENCES event_details(id) ON DELETE CASCADE,
    FOREIGN KEY (uploader_id) REFERENCES user_login(pk_user_id) ON DELETE CASCADE,
    FOREIGN KEY (subscriber_id) REFERENCES event_subscriptions(id) ON DELETE CASCADE
) ENGINE=InnoDB;

-- Event Audit Trail Table
CREATE TABLE event_audit_trail (
    audit_id INT AUTO_INCREMENT PRIMARY KEY,
    table_name VARCHAR(255) NOT NULL,
    record_id INT NOT NULL,
    operation_type ENUM('INSERT', 'UPDATE', 'DELETE') NOT NULL,
    old_values JSON,
    new_values JSON,
    changed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    changed_by INT
) ENGINE=InnoDB;



-- Triggers for Event Details Table
DELIMITER $$
CREATE TRIGGER before_event_details_update
BEFORE UPDATE ON event_details
FOR EACH ROW
BEGIN
    INSERT INTO event_audit_trail (table_name, record_id, operation_type, old_values, new_values, changed_by)
    VALUES (
        'event_details',
        OLD.id,
        'UPDATE',
        JSON_OBJECT('title', OLD.title, 'small_summary', OLD.small_summary, 'rich_explanation', OLD.rich_explanation, 'location', OLD.location, 'latitude', OLD.latitude, 'longitude', OLD.longitude, 'start_datetime', OLD.start_datetime, 'end_datetime', OLD.end_datetime, 'owner_id', OLD.owner_id),
        JSON_OBJECT('title', NEW.title, 'small_summary', NEW.small_summary, 'rich_explanation', NEW.rich_explanation, 'location', NEW.location, 'latitude', NEW.latitude, 'longitude', NEW.longitude, 'start_datetime', NEW.start_datetime, 'end_datetime', NEW.end_datetime, 'owner_id', NEW.owner_id),
        USER()
    );
END$$

CREATE TRIGGER before_event_details_delete
BEFORE DELETE ON event_details
FOR EACH ROW
BEGIN
    INSERT INTO event_audit_trail (table_name, record_id, operation_type, old_values, changed_by)
    VALUES (
        'event_details',
        OLD.id,
        'DELETE',
        JSON_OBJECT('title', OLD.title, 'small_summary', OLD.small_summary, 'rich_explanation', OLD.rich_explanation, 'location', OLD.location, 'latitude', OLD.latitude, 'longitude', OLD.longitude, 'start_datetime', OLD.start_datetime, 'end_datetime', OLD.end_datetime, 'owner_id', OLD.owner_id),
        USER()
    );
END$$

CREATE TRIGGER after_event_details_insert
AFTER INSERT ON event_details
FOR EACH ROW
BEGIN
    INSERT INTO event_audit_trail (table_name, record_id, operation_type, new_values, changed_by)
    VALUES (
        'event_details',
        NEW.id,
        'INSERT',
        JSON_OBJECT('title', NEW.title, 'small_summary', NEW.small_summary, 'rich_explanation', NEW.rich_explanation, 'location', NEW.location, 'latitude', NEW.latitude, 'longitude', NEW.longitude, 'start_datetime', NEW.start_datetime, 'end_datetime', NEW.end_datetime, 'owner_id', NEW.owner_id),
        USER()
    );
END$$

DELIMITER ;

