<?php
    function checkIfDeleted($inputData) {
        $table = $inputData['table'];
        $table = toSnakeCase($table);
        $id = $inputData['id'];
        $db = $inputData['db']['dbLogin'];
        $stmt = $db->prepare("SELECT id FROM $table WHERE id = :id and is_deleted = 1");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($data) {
                        return ['status' => 'failed', 'message' => 'Record already deleted'];
            } else {
                return ['status' => 'success'];
            }
    }
    function getDataForHistory($inputData) {
        $table = $inputData['table'];
        $table = toSnakeCase($table);
        $id = $inputData['id'];
        $db = $inputData['db']['dbLogin'];
        $stmt = $db->prepare("SELECT * FROM $table WHERE id = :id");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        // convert result to json with column names as keys
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        json_encode($data);
        if ($data) {
            return ['status' => 'success', 'historyData' => $data];
        } else {
            return ['status' => 'failed', 'message' => 'Record not found'];
        }
    }
    function checkDuplicateRecord($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $table = $inputData['table'] ?? null;
        $table = toSnakeCase($table);
        $dataKey = extractDataKeyFromTableName($inputData['table']);
        $data = $inputData[$dataKey] ?? [];

        if (!$table) {
            return ['status' => 'failed', 'message' => 'Table not specified'];
        }

        $map = menuTableColumnMapping($table, $inputData['roleId'] ?? 0, "view");
        $columnMap = $map['filtered'] ?? null;

        if (!$columnMap) {
            return ['status' => 'failed', 'message' => 'Invalid table or insufficient permissions'];
        }

        // Define which fields make the record unique (you may want to pass this in $inputData)
        $uniqueFields = $inputData['uniqueFields'];
        $whereClauses = [];
        $params = [];

        foreach ($uniqueFields as $fieldKey) {
            $meta = $columnMap[$fieldKey] ?? null;
            $dbField = $meta['mapping'] ?? null;

            if (!$dbField || !isset($data[$fieldKey])) {
                return ['status' => 'failed', 'message' => "$fieldKey is required"];
            }

            $whereClauses[] = "$dbField = :$fieldKey";
            $params[$fieldKey] = $data[$fieldKey];
        }

        // Add soft delete clause
        $whereClauses[] = "is_deleted = 0";

        $whereSql = implode(' AND ', $whereClauses);
        $sql = "SELECT id FROM $table WHERE $whereSql";
        $stmt = $pdo->prepare($sql);
        foreach ($params as $key => $value) {
            $stmt->bindValue(":$key", $value);
        }
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($result) {
            return ['status' => 'failed', 'message' => 'Record already exists'];
        }

        return ['status' => 'success'];
    } 
    function setTenantId($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $host = $_SERVER['HTTP_HOST'] ?? '';
        $host = strtolower($host);
        $parts = explode('.', $host);
        $cleanParts = array_filter($parts, fn($part) => $part !== 'admin');
        $cleanHost = implode('.', $cleanParts);
        $cleanUrl = 'https://' . $cleanHost;
        // strip www. if it exists
        $cleanUrl = preg_replace('/^https?:\/\/(www\.)?/', 'https://', $cleanUrl);

        // Build SQL
        $sql = "SELECT id FROM tenant WHERE domain = :http and is_deleted = 0";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':http', $cleanUrl);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($data) {
            $_SESSION['tenantId']       = $data['id'];
            return ['status' => 'success', 'message' => 'Tenant ID found'];
        } else {
            return ['status' => 'failed', 'message' => 'Tenant ID not found'];
        }
    }
    function viewAddressWithUserLink($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $userId = $inputData['userId'] ?? null;

        $sql = "select 
                    foa.id as addressId,
                    foa.address_line_1 as addressLine1,
                    foa.address_line_2 as addressLine2,
                    foa.city,
                    foa.county,
                    foa.country,
                    foa.postal_code as postalCode 
                from food_order_address foa
                inner join food_order_user_address foua on foua.address_id = foa.id
                where foua.user_id = :userId
                and is_deleted = 0
                and is_active = 1
                order by is_default desc";

        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':userId', $userId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($data) {
            return ['status' => 'success', 'data' => $data];
        } else {
            return ['status' => 'failed', 'message' => 'Record not found or deleted'];
        }
    }
    function validateAddress($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $addressId = $inputData['basket']['addressId'] ?? null;

        if (!$addressId) {
            return ['status' => 'failed', 'message' => 'Address ID is required'];
        }

        $sql = "SELECT id FROM food_order_address WHERE id = :addressId";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':addressId', $addressId);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($data) {
            return ['status' => 'Found'];
        } else {
            return ['status' => 'Not Found'];
        }
    }
    function checkAddressExists($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $address = $inputData['foodOrderAddress'] ?? [];
        if (empty($address)) {
            return ['status' => 'failed', 'message' => 'Address data is required'];
        }

        $sql = "SELECT id FROM food_order_address WHERE check_sum = :checkSum";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':checkSum', $inputData['foodOrderAddress']['checkSum']);

        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($data) {
            return ['status' => 'Found', 'id' => $data['id']];
        } else {
            return ['status' => 'Not Found'];
        }
    }
    function viewUserAddressList($inputData){
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $userId = $inputData['userId'] ?? null;

        if (!$userId) {
            return ['status' => 'failed', 'message' => 'User ID is required'];
        }

        $sql = "select 
                    foa.id as addressId,
                    foa.address_line_1 as addressLine1,
                    foa.address_line_2 as addressLine2,
                    foa.city,
                    foa.county,
                    foa.country,
                    foa.postal_code as postalCode,
                    foua.is_default as isDefault
                from food_order_address foa
                inner join food_order_user_address foua on foua.address_id = foa.id
                where foua.user_id = :userId
                and foua.is_deleted = 0
                order by foua.is_default desc, foa.id desc";

        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':userId', $userId, PDO::PARAM_INT);
        $stmt->execute();
        $addresses = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if ($addresses) {
            return ['status' => 'success', 'addresses' => $addresses];
        } else {
            return ['status' => 'failed', 'message' => 'No addresses found for this user'];
        }
    }
    function viewAddressById($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $addressId = $inputData['addressId'] ?? null;
        if (!$addressId) {
            return ['status' => 'failed', 'message' => 'Address ID is required'];
        }

        $sql = "SELECT address_line_1, address_line_2, city, county, country, postal_code
                FROM food_order_address
                WHERE id = :addressId";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':addressId', $addressId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($data) {
            return ['status' => 'success', 'data' => $data];
        } else {
            return ['status' => 'failed', 'message' => 'Address not found or deleted'];
        }
    }
    function checkOrderStatus(array $inputData): array{
        $pdo = $inputData['db']['dbApp'];

        $conds  = [];
        $params = [];

        $orderId        = isset($inputData['orderId'])      && $inputData['orderId']        !== null ? (int)$inputData['orderId'] :     null;
        $guestUserId    = isset($inputData['guestUserId'])  && $inputData['guestUserId']    !== null ? (int)$inputData['guestUserId'] : null;
        $userId         = isset($inputData['userId'])       && $inputData['userId']         !== null ? (int)$inputData['userId'] :      null;

        if ($orderId        !== null)       { $conds[] = 'id = :orderId';                $params[':orderId']     = $orderId; }
        if ($guestUserId    !== null)       { $conds[] = 'guest_user_id = :guestUserId'; $params[':guestUserId'] = $guestUserId; }
        if ($userId         !== null)       { $conds[] = 'user_id = :userId';            $params[':userId']      = $userId; }

        if (!$conds) {
            return ['status' => 'failed', 'message' => 'Order ID or Guest/User ID is required'];
        }

        // Parentheses ensure (A OR B OR C) AND 
        $sql = "SELECT id, order_status, payment_status, total_order_price, order_type,
                        total_order_modified_price,  order_full_refund, total_order_modified_reason, created_at, updated_at
                FROM food_order
                WHERE (" . implode(' OR ', $conds) . ")
                " . ($orderId !== null ? " LIMIT 1" : " ORDER BY updated_at DESC LIMIT 1");

        try {
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $data = $stmt->fetch(PDO::FETCH_ASSOC);

            $displayTz = $inputData['tenantTz'] ?? 'Europe/London';

            $data = convertKeysToCamelCase($data);
            $data['totalOrderModifiedPrice'] = number_format($data['totalOrderModifiedPrice'], 2, '.', '');
             // timestamps: DB stores UTC 'Y-m-d H:i:s'
            if (!empty($data['createdAt'])) {
                $rawUtc            = $data['createdAt'];
                $data['createdAt']    = utc_to_iso8601z($rawUtc);                       // e.g. "2025-08-26T22:55:12Z"
                $data['createdAtLocal'] = format_utc_for_tz($rawUtc, $displayTz, 'Y-m-d H:i:s'); // e.g. BST/GMT aware
            } else {
                $data['createdAt']      = null;
                $data['createdAtLocal'] = null;
            }
            if (!empty($data['updatedAt'])) {
                $rawUtc                 = $data['updatedAt'];
                $data['updatedAt']     = utc_to_iso8601z($rawUtc);                       // e.g. "2025-08-26T22:55:12Z"
                $data['updatedAtLocal'] = format_utc_for_tz($rawUtc, $displayTz, 'Y-m-d H:i:s'); // e.g. BST/GMT aware
            } else {
                $data['updatedAt']      = null;
                $data['updatedAtLocal'] = null;
            }
            // if refund remove modified reason
            if ($data['paymentStatus'] === 'refunded') {
                $data['totalOrderModifiedReason'] = '';
                $data['totalOrderModifiedPrice'] = number_format($data['orderFullRefund'], 2, '.', '');
            }
            if (!$data) {
                return ['status' => 'failed', 'message' => 'Order not found or deleted'];
            }
            return ['status' => 'success', 'data' => $data];
        } catch (PDOException $e) {
            return ['status' => 'failed', 'message' => 'Database error', 'error' => $e->getMessage()];
        }
    }
    // login schema check 
    function validateUser($inputData) {
       $userId = $inputData['userId'] ?? null;
        $db = $inputData['db']['dbLogin'] ?? null;

        $stmt = $db->prepare("SELECT fk_user_status_id FROM user_login WHERE pk_user_id = :userId");
        $stmt->bindParam(':userId', $userId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($data) {
            return ['status' => 'Found', 'userStatusId' => $data['fk_user_status_id']];
        } else {
            return ['status' => 'Not Found'];
        }
    }
    // menu schema check
    function checkItemsExist($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $data = [];

        $itemId = $inputData['currentItem']['id'] ?? null; // this will be a single item id
        if (empty($itemId)) {
            return ['status' => 'failed', 'message' => 'Item IDs are required', 'itemId' => $itemId, 'tenantId' => $inputData['tenantId']];
        }

        $sql = "SELECT mi.id, mi.item_name, mi.base_price
                FROM menu_items mi
                inner join menu_category_items mci on mi.id = mci.item_id
                inner join menu_categories mc on mci.category_id = mc.id
                inner join menu_menus m on mc.menu_id = m.id
                WHERE mi.id = :itemId
                    AND m.tenant_id = :tenantId
                   -- AND m.is_active = 1
                    AND m.is_deleted = 0    
                    AND mc.is_active = 1
                    AND mc.is_deleted = 0
                    AND mci.is_active = 1
                    AND mci.is_deleted = 0 
                    AND mi.is_deleted = 0 
                    AND mi.is_active = 1";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':itemId', $itemId, PDO::PARAM_INT);
        $stmt->bindParam(':tenantId', $inputData['tenantId'], PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($data) {
            $res = getItemOptions($inputData);
            if ($res['status'] === 'Not Found') {}else{
                $data['options'] = $res['options'];
            }
            return ['status' => 'Found', 'items' => $data];
        } else {
            return ['status' => 'Not Found', 'message' => 'No items found for the given IDs', 'itemId' => $itemId, 'tenantId' => $inputData['tenantId']];
        }
    }
    function getItemOptions($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $itemId = $inputData['currentItem']['id'] ?? null;
        if (empty($itemId)) {
            return ['status' => 'failed', 'message' => 'Item ID is required'];
        }

        $sql = "SELECT mog.min_select, mog.max_select, mog.is_required, mog.id, moe.extra_name, moj.price_adjustment, moe.id as extra_id
                FROM menu_option_groups mog
                INNER JOIN menu_option_joins moj        ON mog.id = moj.group_id
                INNER JOIN menu_option_extra moe        ON moe.id = moj.extra_id 
                INNER JOIN menu_item_to_options mito    ON mog.id = mito.group_id
                WHERE mito.item_id = :itemId
                AND mito.is_active  = 1
                AND mog.is_active   = 1
                AND moj.is_active   = 1
                AND moe.is_active   = 1
                AND mito.is_deleted = 0
                AND mog.is_deleted  = 0
                AND moj.is_deleted  = 0
                AND moe.is_deleted  = 0";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':itemId', $itemId, PDO::PARAM_INT);
        $stmt->execute();
        $res = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $data = [];

        foreach ($res as $row) {
            $groupId = $row['id'];
            if (!isset($data[$groupId])) {
                $data[$groupId] = [
                    'optionGroupId' => $groupId,
                    'minSelect' => $row['min_select'],
                    'maxSelect' => $row['max_select'],
                    'isRequired' => $row['is_required'],
                    'extra' => []
                ];
            }

            $data[$groupId]['extra'][] = [
                'extraId' => $row['extra_id'],
                'extraName' => $row['extra_name'],
                'priceAdjustment' => $row['price_adjustment']
            ];
        }

        if (!empty($data)) {
            return ['status' => 'Found', 'options' => array_values($data)];
        } else {
            return ['status' => 'Not Found', 'message' => 'No options found for the given item ID'];
        }
    }
    function checkIfOpenForOrders($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $tenantId = $inputData['tenantId'] ?? null;
        if (!$tenantId) return ['status' => 'failed', 'message' => 'Tenant ID is required'];

        $type = strtolower($inputData['basket']['deliveryType'] ?? 'delivery');
        if (!in_array($type, ['delivery', 'collection'], true)) {
            return ['status' => 'failed', 'message' => 'Invalid delivery type. Must be "delivery" or "collection".'];
        }

        $category = $type === 'collection' ? 'store' : $type;

        // Use ONE timezone consistently (swap for per-tenant TZ if you have it)
        $tz = new DateTimeZone('Europe/London');
        $now = new DateTimeImmutable('now', $tz);
        $today       = $now->format('Y-m-d');
        $currentTime = $now->format('H:i:s');
        $weekday     = $now->format('l');

        $sql = "
            SELECT hours_category, open_time, close_time
            FROM tenant_business_hours
            WHERE tenant_id = :tenant_id
            AND is_deleted = 0
            AND is_closed  = 0
            AND (
                    (hours_category = 'special' AND specific_date = :today)
                OR (hours_category = :category   AND opening_day   = :weekday)
            )
            ORDER BY
            CASE WHEN hours_category = 'special' THEN 0 ELSE 1 END,
            CASE
                WHEN (
                (open_time < close_time AND :ct1 BETWEEN open_time AND close_time)
                OR
                (open_time > close_time AND (:ct2 >= open_time OR :ct3 <= close_time))
                ) THEN 0
                ELSE 1
            END,
            open_time
            LIMIT 1
        ";

        $stmt = $pdo->prepare($sql);
        // Bind all placeholders explicitly (unique names for time)
        $stmt->bindValue(':tenant_id',  $tenantId,   PDO::PARAM_INT);
        $stmt->bindValue(':today',      $today,      PDO::PARAM_STR);
        $stmt->bindValue(':category',   $category,   PDO::PARAM_STR);
        $stmt->bindValue(':weekday',    $weekday,    PDO::PARAM_STR);
        $stmt->bindValue(':ct1',        $currentTime, PDO::PARAM_STR);
        $stmt->bindValue(':ct2',        $currentTime, PDO::PARAM_STR);
        $stmt->bindValue(':ct3',        $currentTime, PDO::PARAM_STR);

        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            return ['status' => 'failed', 'isOpen' => false, 'message' => 'No valid business hours found for today'];
        }

        // Compare using SAME TZ
        $open  = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $today . ' ' . $row['open_time'], $tz);
        $close = DateTimeImmutable::createFromFormat('Y-m-d H:i:s', $today . ' ' . $row['close_time'], $tz);

        $isOpen = false;
        if ($row['open_time'] < $row['close_time']) {
            $isOpen = ($now >= $open) && ($now <= $close);
        } else {
            // Overnight, e.g. 22:00 → 02:00
            if ($now >= $open) {
                $close = $close->modify('+1 day');
            }
            $isOpen = ($now >= $open) || ($now <= $close);
        }

        return [
            'status'     => 'success',
            'isOpen'     => $isOpen,
            'openTime'   => $row['open_time'],
            'closeTime'  => $row['close_time'],
            'category'   => $row['hours_category'],
            'now'        => $now->format('Y-m-d H:i:s'),
        ];
    }
    function viewTenantDeliveryRulesBasedOrder(&$inputData){
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $tenantId = $inputData['tenantId'] ?? null;
        if ($tenantId === null) {
            return ['status' => 'failed', 'message' => 'Missing required tenantId'];
        }

        if (!isset($inputData['basket']['order']['totalOrderModifiedPrice'])) {
            return ['status' => 'failed', 'message' => 'Missing required totalOrderModifiedPrice'];
        }

        $totalOrderModifiedPrice = $inputData['basket']['order']['totalOrderModifiedPrice'];

        // Step 1: Try to get rule <= order total
        $sql = "SELECT *
                FROM tenant_delivery_rules
                WHERE tenant_id = :tenantId
                AND is_active = 1
                AND is_deleted = 0
                AND minimum_delivery_amount <= :totalOrderModifiedPrice
                ORDER BY minimum_delivery_amount DESC
                LIMIT 1";

        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':tenantId', $tenantId);
        $stmt->bindParam(':totalOrderModifiedPrice', $totalOrderModifiedPrice);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Step 2: If nothing found, get rule > order total
        if (!$rows) {
            $sqlFallback = "SELECT *
                            FROM tenant_delivery_rules
                            WHERE tenant_id = :tenantId
                            AND is_active = 1
                            AND is_deleted = 0
                            AND minimum_delivery_amount > :totalOrderModifiedPrice
                            ORDER BY minimum_delivery_amount ASC
                            LIMIT 1";
            $stmt = $pdo->prepare($sqlFallback);
            $stmt->bindParam(':tenantId', $tenantId);
            $stmt->bindParam(':totalOrderModifiedPrice', $totalOrderModifiedPrice);
            $stmt->execute();
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }

        if (!$rows) {
            return ['status' => 'failed', 'message' => 'No delivery rules found'];
        }

        // Convert each row
        $deliveryRules = array_map(function($row) {
            return convertKeysToCamelCase(renameIdKeys($row, 'deliveryRulesId'));
        }, $rows);

        // Set the delivery charge to the order structure
        $inputData['basket']['order']['deliveryCharges'] = $rows[0]['delivery_charge'] ?? 0;

        return [
            'status' => 'success',
            'message' => 'Delivery rules found',
            'deliveryRules' => $deliveryRules
        ];
    }
    function viewTenantFees(&$inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $tenantId = $inputData['tenantId'] ?? null;
        if ($tenantId === null) {
            return ['status' => 'failed', 'message' => 'Missing required tenantId'];
        }

        $sql = "SELECT customer_admin_fee, tenant_admin_fee FROM tenant_admin_fees WHERE tenant_id = :tenantId AND is_active = 1 AND is_deleted = 0";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':tenantId', $tenantId);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$rows) {
            return ['status' => 'failed', 'message' => 'No fees found for this tenant'];
        }

        $inputData['basket']['order']['customerAdminFee']   = $rows[0]['customer_admin_fee'] ?? 0;
        $inputData['basket']['order']['tenantAdminFee']     = $rows[0]['tenant_admin_fee'] ?? 0;

        return [
            'status' => 'success',
            'message' => 'Fees found',

        ];
    }
    function viewMinimumOrderAmount(&$inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $tenantId = $inputData['tenantId'] ?? null;
        if ($tenantId === null) {
            return ['status' => 'failed', 'message' => 'Missing required tenantId'];
        }

        $sql = "SELECT minimum_order_amount, small_order_fee FROM tenant WHERE id = :tenantId AND tenant_status = 'active' AND is_deleted = 0";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':tenantId', $tenantId);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if($row){
            $inputData['basket']['order']['minimumOrderAmount'] = $row['minimum_order_amount'] ?? 0;
            $inputData['basket']['order']['smallOrderFee'] = $row['small_order_fee'] ?? 0;
        } else {
            return ['status' => 'failed', 'message' => 'No minimum order amount found for this tenant'];
        }

        return [
            'status' => 'success',
            'minimumOrderAmount' => $row['minimum_order_amount']
        ];
    }
    // tenant schema check
    function viewTenantInfo(&$inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $tenantId = $inputData['tenantId'] ?? null;
        if ($tenantId === null) {
            return ['status' => 'failed', 'message' => 'Missing required tenantId'];
        }

        $sql = "SELECT payment_api_key FROM tenant WHERE id = :tenantId AND is_deleted = 0";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':tenantId', $tenantId);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $inputData['secure']['paymentApiKey'] = $row['payment_api_key'] ?? '';
            return [
                'status' => 'success',
            ];
        } else {
            return ['status' => 'failed', 'message' => 'Tenant not found or deleted'];
        }
    }
    function shopAddress(&$inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $tenantId = $inputData['tenantId'] ?? null;
        if ($tenantId === null) {
            return ['status' => 'failed', 'message' => 'Missing required tenantId'];
        }

        $sql = "SELECT `contact_value`, `label` FROM `tenant_contact` WHERE `tenant_id` = :tenant_id AND is_primary = 'yes' AND is_deleted = 0 AND label = 'Shop Address' ";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':tenant_id', $tenantId);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $inputData['basket']['order']['shopAddress'] = $row;
            return [
                'status' => 'success',
            ];
        } else {
            return ['status' => 'failed', 'message' => "Shop address not found or deleted"];
        }
    }
    // View Order by orderId
    function viewFullOrderDetails($inputData) {
        $db = $inputData['db']['dbApp'];
        $orderId = $inputData['orderId'];
        $tenantId = $inputData['tenantId'];
        $viewLevel = $inputData['viewLevel'] ?? null; 
        $email = $inputData['email'] ?? null;

        if (!isset($orderId)) {
            return ['status' => 'failed', 'message' => 'Order ID is required'];
        }

        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $stmt = $db->prepare("SELECT
                                fo.id,
                                fo.order_status               AS orderStatus,
                                fo.order_type                 AS orderType,
                                fo.total_order_modified_price AS totalOrderModifiedPrice,
                                fo.payment_status             AS paymentStatus,
                                fo.comment,
                                fo.tenant_comment             AS tenantComment,
                                fo.delivery_address_id        AS deliveryAddressId,
                                fo.created_at                 AS createdAt,
                                fo.delivery_fee               AS deliveryFee,
                                fo.order_fee                  AS orderFee,
                                fo.small_order_fee            AS smallOrderFee,
                                COALESCE(ul.user_email, fogu.user_email) AS matchedEmail
                                FROM food_order fo
                                LEFT JOIN food_order_guest_user AS fogu
                                    ON fo.guest_user_id = fogu.id
                                LEFT JOIN user_login AS ul
                                    ON fo.user_id = ul.pk_user_id
                                WHERE fo.id = :order_id
                                AND fo.tenant_id = :tenant_id
                                AND (
                                        (fo.user_id         IS NOT NULL    AND LOWER(ul.user_email)     = LOWER(:email_ul))
                                    OR (fo.guest_user_id    IS NOT NULL    AND LOWER(fogu.user_email)   = LOWER(:email_fogu))
                                    )
                                LIMIT 1;
                                ");
        $stmt->bindValue(':order_id',   $orderId,   PDO::PARAM_INT);
        $stmt->bindValue(':tenant_id',  $tenantId,  PDO::PARAM_INT);
        $stmt->bindValue(':email_ul',  $email,    PDO::PARAM_STR);
        $stmt->bindValue(':email_fogu',$email,    PDO::PARAM_STR);
        $stmt->execute();
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No order details found'];
        }

        $displayTz = $inputData['tenantTz'] ?? 'Europe/London';

        // Convert to camelCase
        foreach ($data as &$order) {
            $order = convertKeysToCamelCase($order);
            $order['totalOrderModifiedPrice'] = number_format($order['totalOrderModifiedPrice'], 2, '.', '');
            if ($viewLevel === 'full') {
                if ($order['orderType'] === 'delivery') {
                    $inputData['deliveryAddressId'] = $order['deliveryAddressId'];
                    $res = viewDeliveryAddress($inputData);
                    if ($res['status'] === 'success') {
                        $order['delivery'] = $res['deliveryAddress'];
                    } else {
                        return ['status' => 'failed', 'message' => 'Delivery address not found'];
                    }
                }else{
                    unset($order['deliveryAddressId']); // else
                }
                $res = getOrderDetails($inputData);
                $order['details'] = $res['status'] === 'success' ? $res['orderDetails'] : "";
                // format deliveryFee, orderFee, smallOrderFee to 2 decimal places
                $order['deliveryFee']   = number_format($order['deliveryFee'], 2, '.', '');
                $order['orderFee']      = number_format($order['orderFee'], 2, '.', '');
                $order['smallOrderFee'] = number_format($order['smallOrderFee'], 2, '.', '');
                //if deliveryFee, orderFee, smallOrderFee > 0, or 0.00 then remove from array
                if ($order['deliveryFee'] == '0.00') {
                    unset($order['deliveryFee']);
                }
                if ($order['orderFee'] == '0.00') {
                    unset($order['orderFee']);
                }
                if ($order['smallOrderFee'] == '0.00') {
                    unset($order['smallOrderFee']);
                }
            }
            // timestamps: DB stores UTC 'Y-m-d H:i:s'
            if (!empty($order['createdAt'])) {
                $rawUtc            = $order['createdAt'];
                $order['createdAt']    = utc_to_iso8601z($rawUtc);                       // e.g. "2025-08-26T22:55:12Z"
                $order['createdAtLocal'] = format_utc_for_tz($rawUtc, $displayTz, 'Y-m-d H:i:s'); // e.g. BST/GMT aware
            } else {
                $order['createdAt']      = null;
                $order['createdAtLocal'] = null;
            }


            
        }   
        return ['status' => 'success', 'orderDetails' => $data];
    }
    function getOrderDetails($inputData) {
        $db = $inputData['db']['dbApp'];
        $orderId = $inputData['orderId'];

        if (!isset($orderId)) {
            return ['status' => 'failed', 'message' => 'Order ID is required'];
        }

        $stmt = $db->prepare("SELECT id, item_name, item_modified_price, item_quantity, item_note, item_status, item_tenant_comment, updated_at
                                FROM food_order_item 
                                WHERE order_id = :order_id");
        $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);
      
        $stmt->execute();
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No order details found'];
        }

        // Convert to camelCase
        foreach ($data as &$detail) {
            // change id to foodOrderItemId
            // if order_type is delivery, add delivery_address_id
            $detail = convertKeysToCamelCase($detail);
            $detail['foodOrderItemId'] = $detail['id'];
            $detail['itemModifiedPrice'] = number_format($detail['itemModifiedPrice'], 2, '.', '');
            $detail['updatedAt'] = date('Y-m-d H:i:s', strtotime($detail['updatedAt']));
            // get extras details
            $inputData['foodOrderItemId'] = $detail['foodOrderItemId'];
            $res = getOrderExtras($inputData);
            $detail['extras'] = $res['status'] === 'success' ? $res['orderDetails'] : [];
            unset($detail['id']); // remove the old id
        }
        
        return ['status' => 'success', 'orderDetails' => $data];
    }
    function getOrderExtras($inputData){
        $db = $inputData['db']['dbApp'];
        $foodOrderItemId = $inputData['foodOrderItemId'];

        if (!isset($foodOrderItemId)) {
            return ['status' => 'failed', 'message' => 'Food Order Item ID is required'];
        }

        $stmt = $db->prepare("SELECT id, extra_name, extra_price, extra_quantity, extra_quantity
                                FROM food_order_extra 
                                WHERE item_id = :food_order_item_id");
        $stmt->bindParam(':food_order_item_id', $foodOrderItemId, PDO::PARAM_INT);
      
        $stmt->execute();
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No order extras found'];
        }

        // Convert to camelCase
        foreach ($data as &$extra) {
            $extra = convertKeysToCamelCase($extra);
            $extra['extraPrice'] = number_format($extra['extraPrice'], 2, '.', '');
            unset($extra['id']); // remove the old id
        }
        
        return ['status' => 'success', 'orderDetails' => $data];
    }
    function viewDeliveryAddress($inputData) {
        $db = $inputData['db']['dbApp'];
        $deliveryAddressId = $inputData['deliveryAddressId'];

        if (!isset($deliveryAddressId)) {
            return ['status' => 'failed', 'message' => 'Delivery Address ID is required'];
        }

        $stmt = $db->prepare("SELECT address_line_1, address_line_2, city, county, country, postal_code FROM food_order_address WHERE id = :id");
        $stmt->bindParam(':id', $deliveryAddressId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No delivery address found for this order'];
        }

       foreach ($data as &$address) {
            //$address = convertKeysToCamelCase($address);
        }

        return ['status' => 'success', 'deliveryAddress' => $data];
    }
    function viewDistanceToAddress($inputData) {
        $db = $inputData['db']['dbApp'];
        $addressId = $inputData['addressId'];

        if (!isset($addressId)) {
            return ['status' => 'failed', 'message' => 'Address ID is required'];
        }

        $stmt = $db->prepare("SELECT distance FROM food_address_to_tenant WHERE address_id = :address_id and tenant_id = :tenant_id");
        $stmt->bindParam(':address_id', $addressId, PDO::PARAM_INT);
        $stmt->bindParam(':tenant_id', $inputData['tenantId'], PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No distance data found for this address'];
        }

        return ['status' => 'success', 'distance' => $data['distance']];
    }
    function checkIfUserToAddressExists(&$inputData){
        $db         = $inputData['db']['dbApp'];
        $userId     = $inputData['foodOrderUserAddress']['userId'];
        $addressId  = $inputData['foodOrderUserAddress']['addressId'];

        if (!isset($userId) || !isset($addressId)) {
            return ['status' => 'failed', 'message' => 'User ID and Address ID are required'];
        }

        $stmt = $db->prepare("SELECT id FROM food_order_user_address WHERE user_id = :user_id AND address_id = :address_id AND is_deleted = 0");
        $stmt->bindParam(':user_id', $userId, PDO::PARAM_INT);
        $stmt->bindParam(':address_id', $addressId, PDO::PARAM_INT);
      
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'not_found', 'message' => 'No address found for this user'];
        }

        return ['status' => 'Found'];
    }
    function viewLatestTerms($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $tenantId = $inputData['tenantId'] ?? null;
        if (!$tenantId) {
            return ['status' => 'failed', 'message' => 'Tenant ID is required'];
        }

        $sql = "SELECT id, terms_content, created_at
                FROM tenant_terms
                WHERE tenant_id = :tenantId
                AND is_active = 1
                AND is_deleted = 0
                ORDER BY created_at DESC
                LIMIT 1";

        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':tenantId', $tenantId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($data) {
            return ['status' => 'success', 'terms' => convertKeysToCamelCase($data)];
        } else {
            return ['status' => 'failed', 'message' => 'No terms found for this tenant'];
        }
    }
?>
