<?php
    function updateMenuTable($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $table = $inputData['table'] ?? null;
        $roleId = $inputData['roleId'] ?? 0;
        $dataKey = $inputData['dataKey'];
        $data = $inputData[$dataKey] ?? [];
        $idField = $inputData['idField'] ?? 'id'; // Default to 'id' unless specified
        $tableName = $inputData['table'] ?? null;
        $table = toSnakeCase($table);
        if (!$table) {
            return ['status' => 'failed', 'message' => 'Table not specified'];
        }

        $map = menuTableColumnMapping($table, $roleId, "update");
        $columnMap = $map['filtered'] ?? null;

        if (!$columnMap) {
            return ['status' => 'failed', 'message' => 'Invalid table or insufficient permissions'];
        }

        $updateParts = [];
        $bindParams = [];
        $errors = [];

        foreach ($columnMap as $key => $meta) {
            // Skip fields with optionalUpdate === null (cannot update)
            if (array_key_exists('optionalUpdate', $meta) && $meta['optionalUpdate'] === null) {
                continue;
            }

            $dbField = $meta['mapping'] ?? null;
            if (!$dbField) {
                continue;
            }

            if (isset($data[$key])) {
                // Check if the user role is allowed to update this field
                if (!in_array($roleId, $meta['updateRole'] ?? [])) {
                    $errors[] = "Role $roleId not allowed to update $key";
                    continue;
                }

                $updateParts[] = "$dbField = :$key";
                $bindParams[$key] = $data[$key];
            } else {
                if (!empty($meta['optionalUpdate']) && $meta['optionalUpdate'] == 1) {
                    // Field is optional for update, OK to skip
                    continue;
                }
                // If field is required and not present, error (optional, depends how strict you want update to be)
            }
        }

        if (!empty($errors)) {
            return [
                'status' => 'failed',
                'message' => implode(', ', $errors)
            ];
        }

        if (empty($updateParts)) {
            return [
                'status' => 'failed',
                'message' => 'No valid fields to update'
            ];
        }

        // if the $updateParts is empty or only contains the idField, we cannot update
        if (count($updateParts) === 1 && in_array("$idField = :id", $updateParts)) {
            return [
                'status' => 'failed',
                'message' => 'No fields to update other than ID'
            ];
        }

        $rowId = $data[$idField] ?? null;
        if ($rowId === null) {
            return [
                'status' => 'failed',
                'message' => "Missing $idField value for WHERE clause"
            ];
        }

        if($table == 'menu_menus'){
            if(isset($data['isActive']) && $data['isActive'] == 1){
                if(in_array('isActive', array_keys($columnMap))){    
                    $tempRes = disableAnyActive($inputData, $table); // empty string as may break update function
                    if($tempRes['status'] == 'failed'){  
                        return ['res' => $tempRes];
                    }
                }
                    
            }
        }
        
        // Build SQL
        $setClause = implode(', ', $updateParts);
        $whereClause = "{$idField} = :id";
        $sql = "UPDATE {$table} SET {$setClause} WHERE {$whereClause}";

        $stmt = $pdo->prepare($sql);

        foreach ($bindParams as $param => $value) {
            $stmt->bindValue(":$param", $value);
        }
        $stmt->bindValue(":id", $rowId);

        $stmt->execute();

        if ($stmt->rowCount() === 0) {
            return [
                'status' => 'failed',
                'message' => "No rows updated. Data may be identical or {$idField} not found."
            ];
        }

        return [
            'status' => 'success',
            'message' => 'Record updated successfully'
        ];
    }
    function deleteContent($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        // Step 1: Validate table, id, and data structure
        $res = validateUpdateInputStep($inputData);
        if ($res['status'] !== 'success') {
            $pdo->rollback();
            return $res;
        }

        $tableName = toSnakeCase($inputData['table']);
        
        //$tableName =   //$inputData['table'];
        $roleId = $inputData['roleId'];
        $dataKey = $inputData['dataKey'];
        $tableData = $inputData[$dataKey];
        $idField = $inputData['idField'];
        $idValue = $inputData['id'];
        $extraSetsSql = "  ";

        // Step 2: Check if record exists and is not already deleted
        $checkSql = "SELECT is_deleted FROM {$tableName} WHERE {$idField} = :id LIMIT 1";
        $checkStmt = $pdo->prepare($checkSql);
        $checkStmt->bindValue(':id', $idValue);
        $checkStmt->execute();
        $record = $checkStmt->fetch(PDO::FETCH_ASSOC);

        if (!$record) {
            $pdo->rollback();
            return ['status' => 'failed', 'message' => "Record not found for ID {$idValue}"];
        }

        if (!empty($record['is_deleted'])) {
            $pdo->rollback();
            return ['status' => 'failed', 'message' => "Record with ID {$idValue} is already deleted"];
        }

        // Step 3: Check user permission to update (delete is an update of is_deleted field)
        $map = menuTableColumnMapping($tableName, $roleId, "update");
        $columnMap = $map['filtered'] ?? null;
        
        if (!$columnMap || !isset($columnMap['isDeleted'])) {
            $pdo->rollback();
            return ['status' => 'failed', 'message' => "Delete permission configuration missing for {$tableName}"];
        }

        if (!in_array($roleId, $columnMap['isDeleted']['updateRole'] ?? [])) {
            $pdo->rollback();
            return ['status' => 'failed', 'message' => "User role not permitted to delete this record"];
        }

        // Step 4: check if there is an is_active field and if it is set to 0 then do delete
        if(in_array('isActive', array_keys($columnMap))){
            $extraSetsSql .= " , is_active = 0 ";
        }

        // Step 5: Perform the soft delete
        $sql = "UPDATE {$tableName} SET is_deleted = 1 {$extraSetsSql} WHERE {$idField} = :id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(":id", $idValue);
        $stmt->execute();

        if ($stmt->rowCount() === 0) {
            $pdo->rollback();
            return ['status' => 'failed', 'message' => "Failed to delete record with ID {$idValue}"];
        }

        return ['status' => 'success', 'message' => 'Record marked as deleted successfully'];
    }
    function disableAnyActive($inputData, $tableName){
        $tenantId = $inputData['tenantId'];
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        // Step 1: check if there are any records that match our condition

        $sql = "select count(*) as count from $tableName where tenant_id = :tenantId and is_deleted = 0";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':tenantId', $tenantId);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        $rowCount = $result['count'];
        if($rowCount == 0){
            return [
                'status' => 'success',
                'message' => 'No rows found.',
                'tenantId' => $tenantId,
                'tableName' => $tableName
            ];
        }else{
            // Step 2: update all records to is_applied = 'no' where is_applied = 'yes' and tenant_id = :tenantId and status = 'draft'
            $sql = "update $tableName set is_active = '0'  where is_active = '1' and tenant_id = :tenantId and is_deleted = 0";
            $stmt = $pdo->prepare($sql);
            $stmt->bindParam(':tenantId', $tenantId);
            $stmt->execute();
            // Step 3: count how many rows are updated
            $rowCount = $stmt->rowCount();
            if ($rowCount === 0) {
                return [
                    'status' => 'Success',
                    'message' => 'No rows updated. None match requirements.'
                ];
            }else{
                // rows updated successfully message
                return [
                    'status' => 'success',
                    'message' => 'Rows updated successfully.',
                    'rowCount' => $rowCount
                ];
            }


            return [
                'status' => 'success',
                'message' => 'Rows found.',
                'rowCount' => $rowCount
            ];
        }
    }
    function askRestaurantToAccept($inputData) {
        // update the payment_intent

        $db = $inputData['db']['dbApp'];
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $paymentIntentId = $inputData['secure']['paymentIntentId'] ?? null;
        $orderId = $inputData['secure']['orderId'] ?? null;
        $tenantId = $inputData['secure']['tenantId'] ?? null;
        
        $sql = "UPDATE food_order 
                SET payment_status = 'payment_pre_auth' 
                WHERE   payment_intent_id = :paymentIntentId 
                    AND id = :orderId 
                    AND tenant_id = :tenantId";
        $stmt = $db->prepare($sql);
        $stmt->bindParam(':paymentIntentId', $paymentIntentId);
        $stmt->bindParam(':orderId', $orderId);
        $stmt->bindParam(':tenantId', $tenantId);
        $stmt->execute();
        if ($stmt->rowCount() === 0) {
            return [
                'status' => 'failed',
                'message' => "Failed to update order status for payment intent {$paymentIntentId} and order {$orderId}"
            ];
        }
        // Notify restaurant (this is a placeholder, implement your notification logic)
        // For example, you might send a message to a queue or trigger a webhook
        // notifyRestaurant($tenantId, $orderId, $paymentIntentId);
        return [
            'status' => 'success'
        ];

    }
    function updateOrderWithPaymentIntent(&$inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $orderId = $inputData['basket']['order']['orderId'] ?? null;
        $paymentIntentId = $inputData['secure']['paymentIntentId'] ?? null;

        if (!$orderId || !$paymentIntentId) {
            return ['status' => 'failed', 'message' => 'Order ID and Payment Intent ID are required'];
        }

        // Update the order with the payment intent ID
        $sql = "UPDATE food_order SET payment_intent_id = :payment_intent_id, payment_status = 'payment_intent' WHERE id = :order_id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':payment_intent_id', $paymentIntentId);
        $stmt->bindValue(':order_id', $orderId);

        if (!$orderId || !$paymentIntentId) {
            return ['status' => 'failed', 'message' => 'Order ID and Payment Intent ID are required'];
        }

        // Update the order with the payment intent ID
        $sql = "UPDATE food_order SET payment_intent_id = :payment_intent_id, payment_status = 'payment_intent' WHERE id = :order_id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':payment_intent_id', $paymentIntentId);
        $stmt->bindValue(':order_id', $orderId);

        try {
            $stmt->execute();
            return ['status' => 'success', 'message' => 'Order updated with payment intent'];
        } catch (PDOException $e) {
            return ['status' => 'failed', 'message' => 'Database error: ' . $e->getMessage()];
        }
    }
    function updateCancelled($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $orderId = $inputData['secure']['orderId'] ?? null;

        if (!$orderId) {
            return ['status' => 'failed', 'message' => 'Order ID is required'];
        }

        // Update the order status to 'cancelled'
        $sql = "UPDATE food_order SET order_status = 'cancelled', payment_status = 'unpaid' WHERE id = :order_id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':order_id', $orderId);
        try {
            $stmt->execute();
            return ['status' => 'success', 'message' => 'Order status updated to cancelled'];
        } catch (PDOException $e) {
            return ['status' => 'failed', 'message' => 'Database error: ' . $e->getMessage()];
        }
    }
    function updatePublicOrderCode($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $orderId = $inputData['basket']['order']['orderId'] ?? null;
        $publicOrderCode = $inputData['basket']['order']['publicOrderCode'] ?? null;

        if (!$orderId || !$publicOrderCode) {
            return ['status' => 'failed', 'message' => 'Order ID and Public Order Code are required'];
        }

        // Update the order with the public order code
        $sql = "UPDATE food_order SET public_order_code = :public_order_code WHERE id = :order_id";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':public_order_code', $publicOrderCode);
        $stmt->bindValue(':order_id', $orderId);

        try {
            $stmt->execute();
            return ['status' => 'success', 'message' => 'Order updated with public order code'];
        } catch (PDOException $e) {
            return ['status' => 'failed', 'message' => 'Database error: ' . $e->getMessage()];
        }
    }
?>