-- ========================================
-- TERMS OF SERVICE SCHEMA FOR FOOD ORDER
-- ========================================

-- Terms of Service Content Table
-- Stores the actual terms content, versioned and tenant-specific
CREATE TABLE food_order_terms_of_service (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tenant_id INT NOT NULL,
    version VARCHAR(50) NOT NULL,              -- e.g., "1.0", "2.1", "2024-01-15"
    title VARCHAR(255) DEFAULT 'Terms of Service',
    content TEXT NOT NULL,                     -- HTML or markdown content
    effective_date DATE NOT NULL,              -- When these terms become active
    expiry_date DATE DEFAULT NULL,             -- Optional: when terms expire
    is_active BOOLEAN DEFAULT FALSE,           -- Only one active version per tenant
    requires_acceptance BOOLEAN DEFAULT TRUE,   -- Some updates might not require re-acceptance
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Ensure only one active version per tenant
    UNIQUE KEY unique_active_per_tenant (tenant_id, is_active),
    -- Ensure unique versions per tenant
    UNIQUE KEY unique_version_per_tenant (tenant_id, version),
    
    INDEX idx_tenant_active (tenant_id, is_active),
    INDEX idx_effective_date (effective_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Terms Acceptance Tracking Table
-- Tracks when users (registered or guest) accept terms
CREATE TABLE food_order_terms_acceptance (
    id INT AUTO_INCREMENT PRIMARY KEY,
    terms_id INT NOT NULL,                     -- Links to specific terms version
    tenant_id INT NOT NULL,
    
    -- User identification (matches food_order table pattern)
    user_id INT DEFAULT NULL,                  -- Registered user
    guest_user_id INT DEFAULT NULL,            -- Guest user
    order_id INT DEFAULT NULL,                 -- Optional: link to specific order
    
    -- Acceptance details
    accepted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    ip_address VARCHAR(45),                    -- IPv4 or IPv6
    user_agent TEXT,                           -- Browser/device info
    acceptance_method ENUM('checkbox', 'button', 'implicit', 'api') DEFAULT 'checkbox',
    
    -- Legal/audit trail
    terms_version VARCHAR(50),                 -- Denormalized for easy reporting
    terms_checksum VARCHAR(64),                -- Hash of terms content when accepted
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (terms_id) REFERENCES food_order_terms_of_service(id),
    FOREIGN KEY (order_id) REFERENCES food_order(id) ON DELETE SET NULL,
    FOREIGN KEY (user_id) REFERENCES user_login(pk_user_id) ON DELETE CASCADE,
    FOREIGN KEY (guest_user_id) REFERENCES food_order_guest_user(id) ON DELETE CASCADE,
    
    -- Ensure user XOR guest (same pattern as food_order table)
    CONSTRAINT chk_terms_user_guest CHECK (
        (user_id IS NOT NULL AND guest_user_id IS NULL) OR
        (user_id IS NULL AND guest_user_id IS NOT NULL)
    ),
    
    INDEX idx_tenant_user (tenant_id, user_id),
    INDEX idx_tenant_guest (tenant_id, guest_user_id),
    INDEX idx_order (order_id),
    INDEX idx_accepted_at (accepted_at),
    INDEX idx_terms_version (tenant_id, terms_version)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;