<?php
    function checkIfDeleted($inputData) {
        $table = $inputData['table'];
        $table = toSnakeCase($table);
        $id = $inputData['id'];
        $db = $inputData['db']['dbApp'];
        $stmt = $db->prepare("SELECT id FROM $table WHERE id = :id and is_deleted = 1");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($data) {
                        return ['status' => 'failed', 'message' => 'Record already deleted'];
            } else {
                return ['status' => 'success'];
            }
    }
    function getDataForHistory($inputData) {
        $table = $inputData['table'];
        $table = toSnakeCase($table);
        $id = $inputData['id'];
        $db = $inputData['db']['dbApp'];
        $stmt = $db->prepare("SELECT * FROM $table WHERE id = :id");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        // convert result to json with column names as keys
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        json_encode($data);
        if ($data) {
            return ['status' => 'success', 'historyData' => $data];
        } else {
            return ['status' => 'failed', 'message' => 'Record not found'];
        }
    }
    function setTenantId($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $host = $_SERVER['HTTP_HOST'] ?? '';
        $host = strtolower($host);
        $parts = explode('.', $host);
        $cleanParts = array_filter($parts, fn($part) => $part !== 'admin');
        $cleanHost = implode('.', $cleanParts);
        $cleanUrl = 'https://' . $cleanHost;

        // Build SQL
        $sql = "SELECT id FROM tenant WHERE domain = :http";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':http', $cleanUrl);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($data) {
            $_SESSION['tenantId']       = $data['id'];
            return ['status' => 'success', 'message' => 'Tenant ID found'];
        } else {
            return ['status' => 'failed', 'message' => 'Tenant ID not found'];
        }
    }
    function viewPreAuthPayments($inputData) {
        $db       = $inputData['db']['dbApp'];
        $tenantId = $inputData['tenantId'] ?? null;

        if (!$tenantId) {
            return ['status' => 'failed', 'message' => 'Tenant ID is required'];
        }

        $sql = "
            SELECT
                id,
                total_order_modified_price AS totalOrderModifiedPrice,
                order_status               AS orderStatus,
                updated_at                 AS updatedAt
            FROM food_order
            WHERE payment_status = 'payment_pre_auth'
            AND tenant_id      = :tenant_id
            ORDER BY updated_at DESC
        ";

        $stmt = $db->prepare($sql);
        $stmt->bindValue(':tenant_id', (int)$tenantId, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$rows) {
            return ['status' => 'failed', 'message' => 'No pre-auth payments found'];
        }

        // Per-tenant/local display zone if you want a human-readable field alongside UTC
        $displayTz = $inputData['tenantTz'] ?? 'Europe/London';

        foreach ($rows as &$r) {
            // price formatting
            $r['totalOrderModifiedPrice'] = number_format((float)$r['totalOrderModifiedPrice'], 2, '.', '');

            // timestamps: DB stores UTC 'Y-m-d H:i:s'
            if (!empty($r['updatedAt'])) {
                $rawUtc            = $r['updatedAt'];
                $r['updatedAt']    = utc_to_iso8601z($rawUtc);                       // e.g. "2025-08-26T22:55:12Z"
                $r['updatedAtLocal'] = format_utc_for_tz($rawUtc, $displayTz, 'Y-m-d H:i:s'); // e.g. BST/GMT aware
            } else {
                $r['updatedAt']      = null;
                $r['updatedAtLocal'] = null;
            }
        }
        unset($r);

        // Group by status
        $grouped = [];
        foreach ($rows as $r) {
            $grouped[$r['orderStatus']][] = convertKeysToCamelCase($r); // keeps your existing convention
        }

        return ['status' => 'success', 'preAuthPayments' => $grouped];
    }
    function viewOrders($inputData) {
        $db = $inputData['db']['dbApp'];
        $tenantId = $inputData['tenantId'];

        $stmt = $db->prepare("
            SELECT `id`, `order_status`, `order_type`, `total_order_modified_price`, `payment_status`, `comment`, `tenant_comment` FROM `food_order` 
                WHERE	`order_status` 	in ('confirmed','cooking','delivery','completed')
                AND 	`payment_status` = 'paid'
                AND 	`tenant_id`		 = :tenant_id
        ");
        $stmt->bindParam(':tenant_id', $tenantId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if (!$data) {
            return ['status' => 'failed', 'message' => 'No orders found'];
        }
        foreach ($data as &$order) {
            $order['totalOrderModifiedPrice'] = number_format($order['totalOrderModifiedPrice'], 2, '.', '');
            // if viewFullOrderDetails is true, add full order details
            if (isset($inputData['viewFullOrderDetails']) && $inputData['viewFullOrderDetails'] === true) {
                $inputData['orderId'] = $order['id'];
                $order['fullOrderDetails'] = viewFullOrderDetails($inputData);
            }
        }

    }
    function viewPaymentIntentOrderById($inputData){
        $db = $inputData['db']['dbApp'];
        $id = $inputData['id'];

        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $stmt = $db->prepare(
            'SELECT payment_intent_id AS paymentIntentId
            FROM food_order
            WHERE id = :id
            LIMIT 1'
        );

        $stmt->bindValue(':id', is_numeric($id) ? (int)$id : (string)$id,
                        is_numeric($id) ? PDO::PARAM_INT : PDO::PARAM_STR);

        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        return $row ? ['status'=>'success','data'=>$row]
                    : ['status'=>'failed','message'=>'Payment Intent not found'];
    }
    function viewFullOrderDetails($inputData) {
        $db = $inputData['db']['dbApp'];
        $orderId = $inputData['orderId'];
        $tenantId = $inputData['tenantId'];
        $viewLevel = $inputData['viewLevel'] ?? null; 
        $receipt = $inputData['receipt'] ?? null;

        if (!isset($orderId)) {
            return ['status' => 'failed', 'message' => 'Order ID is required'];
        }

        $sql_receipt = $receipt === 'enabled' ? " , delivery_fee, order_fee, small_order_fee, user_id, guest_user_id" : "";

        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $stmt = $db->prepare("SELECT `id`, `order_status`, `order_type`, `total_order_modified_price`, `payment_status`, `comment`, `tenant_comment`, `delivery_address_id`, `created_at` $sql_receipt FROM `food_order` 
                                WHERE id = :order_id
                                AND tenant_id = :tenant_id");
        $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);
        $stmt->bindParam(':tenant_id', $tenantId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No order details found'];
        }

        // Convert to camelCase
        foreach ($data as &$order) {
            $order = convertKeysToCamelCase($order);
            $order['totalOrderModifiedPrice'] = number_format($order['totalOrderModifiedPrice'], 2, '.', '');
            if (!empty($order['createdAt'])) {
                $utcYmdHis                  = $order['createdAt'];               // DB value in UTC "Y-m-d H:i:s"
                $order['createdAt']        = utc_to_iso8601z($utcYmdHis);        // "2025-08-26T22:55:12Z" (browser-safe)
                $order['createdAtDisplay'] = format_utc_for_tz(
                    $utcYmdHis,
                    $inputData['tenantTz'] ?? 'Europe/London',
                    'Y-m-d H:i:s'
                ); // e.g. "2025-08-26 23:55:12" during BST
            } else {
                $order['createdAt']        = null;
                $order['createdAtDisplay'] = null;
            }
            if ($viewLevel === 'full') {
                if ($order['orderType'] === 'delivery') {
                    $inputData['deliveryAddressId'] = $order['deliveryAddressId'];
                    $res = viewDeliveryAddress($inputData);
                    if ($res['status'] === 'success') {
                        $order['delivery'] = $res['deliveryAddress'];
                    } else {
                        return ['status' => 'failed', 'message' => 'Delivery address not found'];
                    }
                }else{
                    unset($order['deliveryAddressId']); // else
                }
                $res = getOrderDetails($inputData);
                $order['details'] = $res['status'] === 'success' ? $res['orderDetails'] : "";
            }
            if($receipt === 'enabled'){
                // check if user_id or guest_user_id is set
                if(isset($order['userId']) && !empty($order['userId'])){

                    $table = "";
                    $idName = "id";
                    if(isset($order['guestUserId']) && !empty($order['guestUserId'])){
                        $table = "food_order_guest_user";
                        $idName = "id";
                    }
                    if(isset($order['userId']) && !empty($order['userId'])){
                        $table = "user_login";
                        $idName = "pk_user_id";
                    }

                    $query = "SELECT user_email, f_name, l_name FROM $table WHERE $idName = :user_id LIMIT 1";
                    $stmt = $db->prepare($query);

                    $id = isset($order['guestUserId']) && !empty($order['guestUserId']) ? $order['guestUserId'] : $order['userId'];
                    // validate email if guest user id is set

                    $stmt->bindParam(':user_id', $id, PDO::PARAM_INT);
                    $stmt->execute();
                    $userData = $stmt->fetch(PDO::FETCH_ASSOC);
                    if($userData){
                        $order['userEmail'] = $userData['user_email'] ?? '';
                        $order['firstName'] = $userData['f_name'] ?? '';
                        $order['lastName'] = $userData['l_name'] ?? '';
                    }else{
                        $order['userEmail'] = '';
                        $order['firstName'] = '';
                        $order['lastName'] = '';
                    }
                }
            }
        }   
        return ['status' => 'success', 'orderDetails' => $data];
    }
    function getOrderDetails($inputData) {
        $db = $inputData['db']['dbApp'];
        $orderId = $inputData['orderId'];

        if (!isset($orderId)) {
            return ['status' => 'failed', 'message' => 'Order ID is required'];
        }

        $stmt = $db->prepare("SELECT id, item_name, item_modified_price, item_quantity, item_note, item_status, item_tenant_comment, updated_at
                                FROM food_order_item 
                                WHERE order_id = :order_id");
        $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);
      
        $stmt->execute();
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No order details found'];
        }
        // Optional per-tenant display zone (fallback to London or choose your site-wide default)
        $displayTz = $inputData['tenantTz'] ?? 'Europe/London';

        // Convert to camelCase
        foreach ($data as &$detail) {
            // change id to foodOrderItemId
            // if order_type is delivery, add delivery_address_id
            $detail = convertKeysToCamelCase($detail);
            $detail['foodOrderItemId'] = $detail['id'];
            $detail['itemModifiedPrice'] = number_format($detail['itemModifiedPrice'], 2, '.', '');
            if (!empty($detail['updatedAt'])) {
                $utcYmdHis                  = $detail['updatedAt'];               // DB value in UTC "Y-m-d H:i:s"
                $detail['updatedAt']        = utc_to_iso8601z($utcYmdHis);        // "2025-08-26T22:55:12Z" (browser-safe)
                $detail['updatedAtDisplay'] = format_utc_for_tz(
                    $utcYmdHis,
                    $inputData['tenantTz'] ?? 'Europe/London',
                    'Y-m-d H:i:s'
                ); // e.g. "2025-08-26 23:55:12" during BST
            } else {
                $detail['updatedAt']        = null;
                $detail['updatedAtDisplay'] = null;
            }
            // get extras details
            $inputData['foodOrderItemId'] = $detail['foodOrderItemId'];
            $res = getOrderExtras($inputData);
            $detail['extras'] = $res['status'] === 'success' ? $res['orderDetails'] : [];
            unset($detail['id']); // remove the old id
        }
        
        return ['status' => 'success', 'orderDetails' => $data];
    }
    function getOrderExtras($inputData){
        $db = $inputData['db']['dbApp'];
        $foodOrderItemId = $inputData['foodOrderItemId'];

        if (!isset($foodOrderItemId)) {
            return ['status' => 'failed', 'message' => 'Food Order Item ID is required'];
        }

        $stmt = $db->prepare("SELECT id, extra_name, extra_price, extra_quantity, extra_quantity
                                FROM food_order_extra 
                                WHERE item_id = :food_order_item_id");
        $stmt->bindParam(':food_order_item_id', $foodOrderItemId, PDO::PARAM_INT);
      
        $stmt->execute();
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No order extras found'];
        }

        // Convert to camelCase
        foreach ($data as &$extra) {
            $extra = convertKeysToCamelCase($extra);
            $extra['extraPrice'] = number_format($extra['extraPrice'], 2, '.', '');
            unset($extra['id']); // remove the old id
        }
        
        return ['status' => 'success', 'orderDetails' => $data];
    }
    function viewDeliveryAddress($inputData) {
        $db = $inputData['db']['dbApp'];
        $deliveryAddressId = $inputData['deliveryAddressId'];

        if (!isset($deliveryAddressId)) {
            return ['status' => 'failed', 'message' => 'Delivery Address ID is required'];
        }

        $stmt = $db->prepare("SELECT address_line_1, address_line_2, city, county, country, postal_code FROM food_order_address WHERE id = :id");
        $stmt->bindParam(':id', $deliveryAddressId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No delivery address found for this order'];
        }

       foreach ($data as &$address) {
            //$address = convertKeysToCamelCase($address);
        }

        return ['status' => 'success', 'deliveryAddress' => $data];
    }
    function viewAcceptedOrders($inputData) {
        $db       = $inputData['db']['dbApp'];
        $tenantId = $inputData['tenantId'];

        if (!$tenantId) {
            return ['status' => 'failed', 'message' => 'Tenant ID is required'];
        }

        // ✅ explicit UTC cutoff (2 hours ago)
        $cutoffTime = utc_cutoff_minutes(120); // 'Y-m-d H:i:s' in UTC

        $sql = "
            SELECT
                id,
                total_order_modified_price AS totalOrderModifiedPrice,
                order_status               AS orderStatus,
                updated_at                 AS updatedAt
            FROM food_order
            WHERE payment_status = 'paid'
            AND tenant_id      = :tenant_id
            AND order_status IN ('confirmed', 'cooking', 'delivery', 'completed')
            AND (
                order_status <> 'completed'
                OR updated_at >= :cutoff
            )
            ORDER BY updated_at DESC
        ";

        $stmt = $db->prepare($sql);
        $stmt->bindValue(':tenant_id', (int)$tenantId, PDO::PARAM_INT);
        $stmt->bindValue(':cutoff',    $cutoffTime,    PDO::PARAM_STR);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$rows) {
            return ['status' => 'failed', 'message' => 'No accepted orders found'];
        }

        // Optional: per-tenant display tz for human-readable field
        $displayTz = $inputData['tenantTz'] ?? 'Europe/London';

        foreach ($rows as &$r) {
            // price formatting
            $r['totalOrderModifiedPrice'] = number_format((float)$r['totalOrderModifiedPrice'], 2, '.', '');

            // timestamps: DB stores UTC 'Y-m-d H:i:s'
            if (!empty($r['updatedAt'])) {
                $rawUtc              = $r['updatedAt'];
                $r['updatedAt']      = utc_to_iso8601z($rawUtc); // e.g. "2025-08-26T22:55:12Z"
                $r['updatedAtLocal'] = format_utc_for_tz($rawUtc, $displayTz, 'Y-m-d H:i:s'); // e.g. BST-adjusted
            } else {
                $r['updatedAt']      = null;
                $r['updatedAtLocal'] = null;
            }
        }
        unset($r);

        // group by status
        $grouped = [];
        foreach ($rows as $r) {
            $grouped[$r['orderStatus']][] = $r;
        }

        // convert keys to camelCase (your columns are already camelCase, but this keeps it consistent)
        foreach ($grouped as $status => &$payments) {
            foreach ($payments as &$p) {
                $p = convertKeysToCamelCase($p);
            }
            unset($p);
        }
        unset($payments);

        return ['status' => 'success', 'preAuthPayments' => $grouped];
    }
    function viewTenantInfo($inputData) {
        $db = $inputData['db']['dbApp'];
        $tenantId = $inputData['tenantId'];
        $stmt = $db->prepare("SELECT payment_api_key FROM tenant WHERE id = :tenant_id");
        $stmt->bindParam(':tenant_id', $tenantId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$data) {
            return ['status' => 'failed', 'message' => 'Tenant not found'];
        }
        $data = convertKeysToCamelCase($data);
        $data['paymentApiKey'] = $data['paymentApiKey'] ?? '';
        return ['status' => 'success', 'tenantInfo' => $data];
    }

?>