<?php
    function checkIfDeleted($inputData) {
        $table = $inputData['table'];
        $table = toSnakeCase($table);
        $id = $inputData['id'];
        $db = $inputData['db']['dbApp'];
        $stmt = $db->prepare("SELECT id FROM $table WHERE id = :id and is_deleted = 1");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
            if ($data) {
                        return ['status' => 'failed', 'message' => 'Record already deleted'];
            } else {
                return ['status' => 'success'];
            }
    }
    function getDataForHistory($inputData) {
        $table = $inputData['table'];
        $table = toSnakeCase($table);
        $id = $inputData['id'];
        $db = $inputData['db']['dbApp'];
        $stmt = $db->prepare("SELECT * FROM $table WHERE id = :id");
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->execute();
        // convert result to json with column names as keys
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        json_encode($data);
        if ($data) {
            return ['status' => 'success', 'historyData' => $data];
        } else {
            return ['status' => 'failed', 'message' => 'Record not found'];
        }
    }
    function setTenantId($inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $host = $_SERVER['HTTP_HOST'] ?? '';
        $host = strtolower($host);
        $parts = explode('.', $host);
        $cleanParts = array_filter($parts, fn($part) => $part !== 'admin');
        $cleanHost = implode('.', $cleanParts);
        $cleanUrl = 'https://' . $cleanHost;
        // strip www. if it exists
        $cleanUrl = preg_replace('/^https?:\/\/(www\.)?/', 'https://', $cleanUrl);

        // Build SQL
        $sql = "SELECT id FROM tenant WHERE domain = :http and is_deleted = 0";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':http', $cleanUrl);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($data) {
            $_SESSION['tenantId']       = $data['id'];
            return ['status' => 'success', 'message' => 'Tenant ID found'];
        } else {
            return ['status' => 'failed', 'message' => 'Tenant ID not found'];
        }
    }
    function viewPreAuthPayments($inputData) {
        $db       = $inputData['db']['dbApp'];
        $tenantId = $inputData['tenantId'] ?? null;

        if (!$tenantId) {
            return ['status' => 'failed', 'message' => 'Tenant ID is required'];
        }

        $sql = "
            SELECT
                id,
                total_order_modified_price AS totalOrderModifiedPrice,
                order_status               AS orderStatus,
                updated_at                 AS updatedAt
            FROM food_order
            WHERE payment_status = 'payment_pre_auth'
            AND tenant_id      = :tenant_id
            ORDER BY updated_at DESC
        ";

        $stmt = $db->prepare($sql);
        $stmt->bindValue(':tenant_id', (int)$tenantId, PDO::PARAM_INT);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$rows) {
            return ['status' => 'failed', 'message' => 'No pre-auth payments found'];
        }

        // Per-tenant/local display zone if you want a human-readable field alongside UTC
        $displayTz = $inputData['tenantTz'] ?? 'Europe/London';

        foreach ($rows as &$r) {
            // price formatting
            $r['totalOrderModifiedPrice'] = number_format((float)$r['totalOrderModifiedPrice'], 2, '.', '');

            // timestamps: DB stores UTC 'Y-m-d H:i:s'
            if (!empty($r['updatedAt'])) {
                $rawUtc            = $r['updatedAt'];
                $r['updatedAt']    = utc_to_iso8601z($rawUtc);                       // e.g. "2025-08-26T22:55:12Z"
                $r['updatedAtLocal'] = format_utc_for_tz($rawUtc, $displayTz, 'Y-m-d H:i:s'); // e.g. BST/GMT aware
            } else {
                $r['updatedAt']      = null;
                $r['updatedAtLocal'] = null;
            }
        }
        unset($r);

        // Group by status
        $grouped = [];
        foreach ($rows as $r) {
            $grouped[$r['orderStatus']][] = convertKeysToCamelCase($r); // keeps your existing convention
        }

        return ['status' => 'success', 'preAuthPayments' => $grouped];
    }
    function viewOrders($inputData) {
        $db = $inputData['db']['dbApp'];
        $tenantId = $inputData['tenantId'];

        $stmt = $db->prepare("
            SELECT `id`, `order_status`, `order_type`, `total_order_modified_price`, `payment_status`, `comment`, `tenant_comment` FROM `food_order` 
                WHERE	`order_status` 	in ('confirmed','cooking','delivery','completed')
                AND 	`payment_status` = 'paid'
                AND 	`tenant_id`		 = :tenant_id
        ");
        $stmt->bindParam(':tenant_id', $tenantId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if (!$data) {
            return ['status' => 'failed', 'message' => 'No orders found'];
        }
        foreach ($data as &$order) {
            $order['totalOrderModifiedPrice'] = number_format($order['totalOrderModifiedPrice'], 2, '.', '');
            // if viewFullOrderDetails is true, add full order details
            if (isset($inputData['viewFullOrderDetails']) && $inputData['viewFullOrderDetails'] === true) {
                $inputData['orderId'] = $order['id'];
                $order['fullOrderDetails'] = viewFullOrderDetails($inputData);
            }
        }

    }
    function viewPaymentIntentOrderById($inputData){
        $db = $inputData['db']['dbApp'];
        $id = $inputData['id'];

        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $stmt = $db->prepare(
            'SELECT payment_intent_id AS paymentIntentId
            FROM food_order
            WHERE id = :id
            LIMIT 1'
        );

        $stmt->bindValue(':id', is_numeric($id) ? (int)$id : (string)$id,
                        is_numeric($id) ? PDO::PARAM_INT : PDO::PARAM_STR);

        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        return $row ? ['status'=>'success','data'=>$row]
                    : ['status'=>'failed','message'=>'Payment Intent not found'];
    }
    function viewFullOrderDetails($inputData) {
        $db        = $inputData['db']['dbApp'];
        $orderId   = isset($inputData['orderId'])  ? (int)$inputData['orderId']  : null;
        $tenantId  = isset($inputData['tenantId']) ? (int)$inputData['tenantId'] : null;
        $viewLevel = $inputData['viewLevel']  ?? null;       // 'full' to include details/address
        $receipt   = $inputData['receipt']    ?? null;       // 'enabled' to include fees + user fields
        $tenantTz  = $inputData['tenantTz']   ?? 'Europe/London';

        if (!$orderId)  return ['status' => 'failed', 'message' => 'Order ID is required'];
        if (!$tenantId) return ['status' => 'failed', 'message' => 'Tenant ID is required'];

        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        // Base SELECT
        $select =
            "SELECT fo.`id`,
                    fo.`order_status`,
                    fo.`order_type`,
                    fo.`total_order_modified_price`,
                    fo.`payment_status`,
                    fo.`comment`,
                    fo.`tenant_comment`,
                    fo.`delivery_address_id`,
                    fo.`created_at`";
        $joins = "";
        $tail  = "WHERE fo.id = :order_id AND fo.tenant_id = :tenant_id LIMIT 1";  


        if ($receipt === 'enabled') {
            // Add fee columns and user/guest lookup (both LEFT JOINed; prefer user_login over guest if both exist)
            $select .= ",
                    fo.`delivery_fee`,
                    fo.`order_fee`,
                    fo.`small_order_fee`,
                    fo.`user_id`,
                    fo.`guest_user_id`,
                    COALESCE(ul.`user_email`, g.`user_email`)   AS `user_email`,
                    COALESCE(ul.`f_name`, g.`f_name`)           AS `f_name`,
                    COALESCE(ul.`l_name`, g.`l_name`)           AS `l_name`";

            $joins  = "
                LEFT JOIN `user_login` ul
                    ON ul.`pk_user_id` = fo.`user_id`
                LEFT JOIN `food_order_guest_user` g
                    ON g.`id` = fo.`guest_user_id`";
        }

        $sql = "$select
                FROM `food_order` fo
                $joins
                $tail";
        
        $stmt = $db->prepare($sql);
        $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);
        $stmt->bindParam(':tenant_id', $tenantId, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            return ['status' => 'failed', 'message' => 'No order details found'];
        }

        // ---- Transform keys & amounts
        $order = convertKeysToCamelCase($row);

        // Always format price consistently (string w/ 2dp)
        if (isset($order['totalOrderModifiedPrice'])) {
            $order['totalOrderModifiedPrice'] = number_format((float)$order['totalOrderModifiedPrice'], 2, '.', '');
        }

        // ---- createdAt handling (DB stored as UTC "Y-m-d H:i:s")
        if (!empty($order['createdAt'])) {
            $utcYmdHis              = $order['createdAt'];
            $order['createdAt']     = utc_to_iso8601z($utcYmdHis);                  // e.g. "2025-08-26T22:55:12Z"
            $order['createdAtDisplay'] = format_utc_for_tz($utcYmdHis, $tenantTz, 'Y-m-d H:i:s'); // tenant-local string
        } else {
            $order['createdAt']        = null;
            $order['createdAtDisplay'] = null;
        }

        // ---- Delivery address (only when needed)
        if ($viewLevel === 'full') {
            if (($order['orderType'] ?? null) === 'delivery' && !empty($order['deliveryAddressId'])) {
                // add required param for the address function
                $inputData['deliveryAddressId'] = (int)$order['deliveryAddressId'];
                $res = viewDeliveryAddress($inputData);
                if ($res['status'] === 'success') {
                    $order['delivery'] = $res['deliveryAddress'];
                } else {
                    return ['status' => 'failed', 'message' => 'Delivery address not found'];
                }
            } else {
                // not delivery; don’t leak this internal id
                unset($order['deliveryAddressId']);
            }

            // line items
            $res = getOrderDetails($inputData);
            $order['details'] = ($res['status'] === 'success') ? ($res['orderDetails'] ?? []) : [];
        } else {
            // if not full, we do not return details; hide internal address id for cleanliness
            unset($order['deliveryAddressId']);
        }

        // ---- Receipt enrichment (fees + user fields)
        if ($receipt === 'enabled') {
            // Normalise user fields even if JOINs returned null
            $order['userEmail'] = $order['userEmail'] ?? ($order['user_email'] ?? '');
            $order['firstName'] = $order['firstName'] ?? ($order['fName'] ?? '');
            $order['lastName']  = $order['lastName']  ?? ($order['lName'] ?? '');
            // Clean up alternative casings brought by convertKeysToCamelCase
            unset($order['user_email'], $order['fName'], $order['lName']);
        } else {
            // If not building a receipt, strip fee/user identifiers for safety
            unset($order['deliveryFee'], $order['orderFee'], $order['smallOrderFee'], $order['userId'], $order['guestUserId']);
        }

        // Wrap as array to preserve your external shape: orderDetails => [ … ]
        $payload = [$order];

        // Optional: comment this out in production to avoid leaking PII in logs
        // error_log(print_r($payload, true));

        return ['status' => 'success', 'orderDetails' => $payload];
    }
    function getOrderDetails($inputData) {
        $db = $inputData['db']['dbApp'];
        $orderId = $inputData['orderId'];

        if (!isset($orderId)) {
            return ['status' => 'failed', 'message' => 'Order ID is required'];
        }
        $sql = "SELECT id, item_name, item_modified_price, item_quantity, item_note, item_status, item_tenant_comment, updated_at
                                FROM food_order_item 
                                WHERE order_id = :order_id AND NOT (item_quantity = 0 AND (item_status = 'partial_refund' or item_status = 'refunded'))";

        $stmt = $db->prepare($sql);
        $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);

        $stmt->execute();
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No order details found'];
        }
        // Optional per-tenant display zone (fallback to London or choose your site-wide default)
        $displayTz = $inputData['tenantTz'] ?? 'Europe/London';

        // Convert to camelCase
        foreach ($data as &$detail) {
            // change id to foodOrderItemId
            // if order_type is delivery, add delivery_address_id
            $detail = convertKeysToCamelCase($detail);
            $detail['foodOrderItemId'] = $detail['id'];
            $detail['itemModifiedPrice'] = number_format($detail['itemModifiedPrice'], 2, '.', '');
            if (!empty($detail['updatedAt'])) {
                $utcYmdHis                  = $detail['updatedAt'];               // DB value in UTC "Y-m-d H:i:s"
                $detail['updatedAt']        = utc_to_iso8601z($utcYmdHis);        // "2025-08-26T22:55:12Z" (browser-safe)
                $detail['updatedAtDisplay'] = format_utc_for_tz(
                    $utcYmdHis,
                    $inputData['tenantTz'] ?? 'Europe/London',
                    'Y-m-d H:i:s'
                ); // e.g. "2025-08-26 23:55:12" during BST
            } else {
                $detail['updatedAt']        = null;
                $detail['updatedAtDisplay'] = null;
            }
            // get extras details
            $inputData['foodOrderItemId'] = $detail['foodOrderItemId'];
            $res = getOrderExtras($inputData);
            $detail['extras'] = $res['status'] === 'success' ? $res['orderDetails'] : [];
            unset($detail['id']); // remove the old id
        }
        
        return ['status' => 'success', 'orderDetails' => $data];
    }
    function getOrderExtras($inputData){
        $db = $inputData['db']['dbApp'];
        $foodOrderItemId = $inputData['foodOrderItemId'];

        if (!isset($foodOrderItemId)) {
            return ['status' => 'failed', 'message' => 'Food Order Item ID is required'];
        }

        $stmt = $db->prepare("SELECT id, extra_name, extra_price, extra_quantity, extra_quantity
                                FROM food_order_extra 
                                WHERE item_id = :food_order_item_id");
        $stmt->bindParam(':food_order_item_id', $foodOrderItemId, PDO::PARAM_INT);
      
        $stmt->execute();
        $data = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No order extras found'];
        }

        // Convert to camelCase
        foreach ($data as &$extra) {
            $extra = convertKeysToCamelCase($extra);
            $extra['extraPrice'] = number_format($extra['extraPrice'], 2, '.', '');
            unset($extra['id']); // remove the old id
        }
        
        return ['status' => 'success', 'orderDetails' => $data];
    }
    function viewDeliveryAddress($inputData) {
        $db = $inputData['db']['dbApp'];
        $deliveryAddressId = $inputData['deliveryAddressId'];

        if (!isset($deliveryAddressId)) {
            return ['status' => 'failed', 'message' => 'Delivery Address ID is required'];
        }

        $stmt = $db->prepare("SELECT address_line_1, address_line_2, city, county, country, postal_code FROM food_order_address WHERE id = :id");
        $stmt->bindParam(':id', $deliveryAddressId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No delivery address found for this order'];
        }

       foreach ($data as &$address) {
            //$address = convertKeysToCamelCase($address);
        }

        return ['status' => 'success', 'deliveryAddress' => $data];
    }
    function viewAcceptedOrders($inputData) {
        $db       = $inputData['db']['dbApp'];
        $tenantId = $inputData['tenantId'];

        if (!$tenantId) {
            return ['status' => 'failed', 'message' => 'Tenant ID is required'];
        }

        // ✅ explicit UTC cutoff (2 hours ago)
        $cutoffTime = utc_cutoff_minutes(120); // 'Y-m-d H:i:s' in UTC

        $sql = "
            SELECT
                id,
                total_order_modified_price AS totalOrderModifiedPrice,
                order_status               AS orderStatus,
                updated_at                 AS updatedAt, 
                order_type                 AS orderType
            FROM food_order
            WHERE payment_status IN ('paid', 'partial_refund')
            AND tenant_id      = :tenant_id
            AND order_status IN ('confirmed', 'cooking', 'delivery', 'completed' )
            AND NOT (payment_status = 'partial_refund' AND total_order_modified_price = 0.00)
            AND (
                order_status <> 'completed'
                OR updated_at >= :cutoff
            )
            ORDER BY updated_at DESC
        ";

        $stmt = $db->prepare($sql);
        $stmt->bindValue(':tenant_id', (int)$tenantId, PDO::PARAM_INT);
        $stmt->bindValue(':cutoff',    $cutoffTime,    PDO::PARAM_STR);
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

        if (!$rows) {
            return ['status' => 'failed', 'message' => 'No accepted orders found'];
        }

        // Optional: per-tenant display tz for human-readable field
        $displayTz = $inputData['tenantTz'] ?? 'Europe/London';

        foreach ($rows as &$r) {
            // price formatting
            $r['totalOrderModifiedPrice'] = number_format((float)$r['totalOrderModifiedPrice'], 2, '.', '');

            // timestamps: DB stores UTC 'Y-m-d H:i:s'
            if (!empty($r['updatedAt'])) {
                $rawUtc              = $r['updatedAt'];
                $r['updatedAt']      = utc_to_iso8601z($rawUtc); // e.g. "2025-08-26T22:55:12Z"
                $r['updatedAtLocal'] = format_utc_for_tz($rawUtc, $displayTz, 'Y-m-d H:i:s'); // e.g. BST-adjusted
            } else {
                $r['updatedAt']      = null;
                $r['updatedAtLocal'] = null;
            }
        }
        unset($r);

        // group by status
        $grouped = [];
        foreach ($rows as $r) {
            $grouped[$r['orderStatus']][] = $r;
        }

        // convert keys to camelCase (your columns are already camelCase, but this keeps it consistent)
        foreach ($grouped as $status => &$payments) {
            foreach ($payments as &$p) {
                $p = convertKeysToCamelCase($p);
            }
            unset($p);
        }
        unset($payments);

        return ['status' => 'success', 'preAuthPayments' => $grouped];
    }
    function viewTenantInfo($inputData) {
        $db = $inputData['db']['dbApp'];
        $tenantId = $inputData['tenantId'];
        $stmt = $db->prepare("SELECT payment_api_key FROM tenant WHERE id = :tenant_id");
        $stmt->bindParam(':tenant_id', $tenantId, PDO::PARAM_INT);
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$data) {
            return ['status' => 'failed', 'message' => 'Tenant not found'];
        }
        $data = convertKeysToCamelCase($data);
        $data['paymentApiKey'] = $data['paymentApiKey'] ?? '';
        return ['status' => 'success', 'tenantInfo' => $data];
    }
    function getTenantDetails(&$inputData) {
        $tenantId = $inputData['tenantId'] ?? null;
        if (!$tenantId) {
            return ['status' => 'failed', 'message' => 'Tenant ID is required'];
        }

        $db        = $inputData['db']['dbApp'];
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $sql = "SELECT company_name as companyName, company_number as companyNumber FROM `tenant` WHERE `id` = :tenant_id LIMIT 1";
        $stmt = $db->prepare($sql);
        $stmt->bindParam(':tenant_id', $tenantId, PDO::PARAM_INT);
        $stmt->execute();
        $tenant = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$tenant) {
            return ['status' => 'failed', 'message' => 'Tenant not found'];
        }

        $inputData['tenantDetails'] = $tenant;

        return ['status' => 'success', 'tenantDetails' => $tenant];
    }
    function shopAddress(&$inputData) {
        $pdo = $inputData['db']['dbApp'];
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

        $tenantId = $inputData['tenantId'] ?? null;
        if ($tenantId === null) {
            return ['status' => 'failed', 'message' => 'Missing required tenantId'];
        }

        $sql = "SELECT `contact_value`, `label` FROM `tenant_contact` WHERE `tenant_id` = :tenant_id AND is_primary = 'yes' AND is_deleted = 0 AND label = 'Shop Address' ";
        $stmt = $pdo->prepare($sql);
        $stmt->bindParam(':tenant_id', $tenantId);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $inputData['shopAddress'] = $row;
            return [
                'status' => 'success',
                'message' => 'Shop address found',
                'data' => $row
            ];
        } else {
            return ['status' => 'failed', 'message' => "Shop address not found or deleted"];
        }
    }
    function viewCurrentOrderAmounts(&$inputData) {
        $db = $inputData['db']['dbApp'];
        $orderId = $inputData['orderId'];

        if (!isset($orderId)) {
            return ['status' => 'failed', 'message' => 'Order ID is required'];
        }

        $stmt = $db->prepare("SELECT total_order_modified_price FROM food_order WHERE id = :order_id");
        $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);
      
        $stmt->execute();
        $data = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$data) {
            return ['status' => 'failed', 'message' => 'No order found'];
        }

        $inputData['currentOrderAmount'] = number_format((float)$data['total_order_modified_price'], 2, '.', '');

        return ['status' => 'success'];
    }
    function validateOrderItemForRefund(&$inputData) {
        $db      = $inputData['db']['dbApp'];
        $orderId = (int)($inputData['itemValidation']['orderId'] ?? 0);
        $itemId  = (int)($inputData['itemValidation']['id']      ?? 0);
        $qty     = (int)($inputData['itemValidation']['qty']     ?? 0);

        if ($orderId <= 0 || $itemId <= 0 || $qty <= 0) {
            return ['status' => 'failed', 'message' => 'Order ID, Item ID and Quantity must be positive integers'];
        }

        $stmt = $db->prepare("
            SELECT item_quantity, item_modified_price, item_base_price, item_modified_reason, item_name
            FROM food_order_item
            WHERE order_id = :order_id AND id = :item_id
            LIMIT 1
        ");
        $stmt->execute([':order_id' => $orderId, ':item_id' => $itemId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            return ['status' => 'failed', 'message' => "Item ID $itemId not found in Order ID $orderId"];
        }

        $orderedQty = (int)$row['item_quantity'];
        $baseMinor  = (int)round(((float)$row['item_base_price']) * 100);          // per-unit price in pence
        $remainMinor= isset($row['item_modified_price'])
                        ? max(0, (int)round(((float)$row['item_modified_price']) * 100)) // remaining in pence
                        : ($orderedQty * $baseMinor);                                // default full line value

        $reason = (string)($row['item_modified_reason'] ?? '');

        if ($qty > $orderedQty) {
            return ['status' => 'failed', 'message' => "Requested qty ($qty) exceeds ordered qty ($orderedQty) for Item ID $itemId"];
        }

        $maxRefundableQty = intdiv($remainMinor, $baseMinor);
        if ($maxRefundableQty <= 0) {
            return ['status' => 'failed', 'message' => "No refundable balance remains for Item ID $itemId"];
        }
        if ($qty > $maxRefundableQty) {
            return ['status' => 'failed', 'message' => "Requested qty ($qty) exceeds refundable qty ($maxRefundableQty) for Item ID $itemId"];
        }

        $refundMinor      = $qty * $baseMinor;

        // ✅ overwrite the same keys, but now in *pence*
                $inputData['itemValidation']['itemModifiedPrice']   = $remainMinor;    // remaining refundable, pence
                $inputData['itemValidation']['orderedQty']          = $orderedQty;
                $inputData['itemValidation']['itemBasePrice']       = $baseMinor;        // per unit, pence
                $inputData['itemValidation']['itemModifiedReason']  = $reason;
                $inputData['itemValidation']['itemName']            = $row['item_name'] ?? '';
                $inputData['itemValidation']['refundQty']           = $qty;

        return ['status' => 'success'];
    }
    function viewOrderContact(&$inputData){
        $db        = $inputData['db']['dbApp'];
        $orderId   = isset($inputData['orderId'])  ? (int)$inputData['orderId']  : null;
        $tenantId  = isset($inputData['tenantId']) ? (int)$inputData['tenantId'] : null;

        // Add fee columns and user/guest lookup (both LEFT JOINed; prefer user_login over guest if both exist)
        $select = "
                SELECT
                fo.`id`,
                COALESCE(ul.`user_email`, g.`user_email`)   AS `user_email`,
                COALESCE(ul.`f_name`, g.`f_name`)           AS `f_name`,
                COALESCE(ul.`l_name`, g.`l_name`)           AS `l_name`";
        $joins  = "
            LEFT JOIN `user_login` ul
                ON ul.`pk_user_id` = fo.`user_id`
            LEFT JOIN `food_order_guest_user` g
                ON g.`id` = fo.`guest_user_id`";

        $tail  = " WHERE fo.id = :order_id AND fo.tenant_id = :tenant_id LIMIT 1";

        $sql = "$select
                FROM `food_order` fo
                $joins
                $tail";

        $stmt = $db->prepare($sql);
        $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);
        $stmt->bindParam(':tenant_id', $tenantId, PDO::PARAM_INT);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) {
            return ['status' => 'failed', 'message' => 'No order details found'];
        }
        $order = convertKeysToCamelCase($row);
        $order['id'] = $order['id'] ?? 0;
        $order['userEmail'] = $order['userEmail'] ?? ($order['user_email'] ?? '');
        $order['firstName'] = $order['firstName'] ?? ($order['fName'] ?? '');
        $order['lastName']  = $order['lastName']  ?? ($order['lName'] ?? '');
        // if refund type is full, format totalOrderModifiedPrice
        unset($order['user_email'], $order['fName'], $order['lName']);  
        $inputData['order'] = $order;
        return ['status' => 'success']; 
    }
    function viewFullOrderListPaginated($inputData){
        // we will have options for the lenght of time eg start and end date, or how many to show per page, and where to start
        $db             = $inputData['db']['dbApp'];
        $tenantId       = $inputData['tenantId'];
        $startDate      = isset($inputData['startDate'])        ? $inputData['startDate'] : null;
        $endDate        = isset($inputData['endDate'])          ? $inputData['endDate'] : null;
        $limit          = isset($inputData['limit'])            && is_numeric($inputData['limit']) ? (int)$inputData['limit'] : 25;
        $offset         = isset($inputData['offset'])           && is_numeric($inputData['offset']) ? (int)$inputData['offset'] : 0;
        $orderByValue   = isset($inputData['orderBy'])          && in_array(strtoupper($inputData['orderBy']), ['ASC', 'DESC']) ? strtoupper($inputData['orderBy']) : 'DESC';       
        $orderType      = isset($inputData['orderType'])        && in_array(strtoupper($inputData['orderType']), ['DELIVERY', 'PICKUP', 'DINE_IN', 'COLLECTION']) ? strtoupper($inputData['orderType']) : null; // not used yet
        $orderStatus    = isset($inputData['orderStatus'])      && in_array(strtoupper($inputData['orderStatus']), ['PENDING', 'CONFIRMED', 'COOKING', 'DELIVERY', 'COMPLETED', 'CANCELLED']) ? strtoupper($inputData['orderStatus']) : null; // not used yet
        $paymentStatus  = isset($inputData['paymentStatus'])    && in_array(strtoupper($inputData['paymentStatus']), ['UNPAID', 'PAYMENT_INTENT', 'PAID', 'REFUNDED', 'PARTIAL_REFUND', 'PAYMENT_PRE_AUTH']) ? strtoupper($inputData['paymentStatus']) : null; // not used yet
        if (!$tenantId) {
            return ['status' => 'failed', 'message' => 'Tenant ID is required'];
        }
        $where = "WHERE fo.tenant_id = :tenant_id";
        $orderBy = "ORDER BY fo.id $orderByValue";
        
        // build where clause based on filters
        if($startDate !== null && $endDate !== null){
            $where .= " AND fo.created_at BETWEEN :start_date AND :end_date";
        } elseif ($startDate !== null) {
            $where .= " AND fo.created_at >= :start_date";
        } elseif ($endDate !== null) {
            $where .= " AND fo.created_at <= :end_date";
        }
        // if orderType is set, add to where clause
        if($orderType !== null){
            $where .= " AND UPPER(fo.order_type) = :order_type";
        }
        // if orderStatus is set, add to where clause
        if($orderStatus !== null){
            $where .= " AND UPPER(fo.order_status) = :order_status";
        }
        // if paymentStatus is set, add to where clause
        if($paymentStatus !== null){
            $where .= " AND UPPER(fo.payment_status) = :payment_status";
        }
        
        $sql = "SELECT fo.id, fo.order_status, fo.order_type, fo.total_order_modified_price, fo.payment_status, fo.comment, fo.tenant_comment, fo.created_at
                FROM food_order fo
                $where
                $orderBy
                LIMIT :limit OFFSET :offset";

        $sqlCount = "SELECT COUNT(*) as total
                FROM food_order fo
                $where";
        
        // Get total count for pagination
        $stmtCount = $db->prepare($sqlCount);
        $stmtCount->bindParam(':tenant_id', $tenantId, PDO::PARAM_INT);
        if($startDate !== null){
            $startDateStr = date('Y-m-d H:i:s', $startDate);
            $stmtCount->bindParam(':start_date', $startDateStr, PDO::PARAM_STR);
        }
        if($endDate !== null){
            $endDateStr = date('Y-m-d H:i:s', $endDate);
            $stmtCount->bindParam(':end_date', $endDateStr, PDO::PARAM_STR);
        }
        if($orderType !== null){
            $stmtCount->bindParam(':order_type', $orderType, PDO::PARAM_STR);
        }
        if($orderStatus !== null){
            $stmtCount->bindParam(':order_status', $orderStatus, PDO::PARAM_STR);
        }
        if($paymentStatus !== null){
            $stmtCount->bindParam(':payment_status', $paymentStatus, PDO::PARAM_STR);
        }

        $stmtCount->execute();
        $countResult = $stmtCount->fetch(PDO::FETCH_ASSOC);

        $stmt = $db->prepare($sql);
        $stmt->bindParam(':tenant_id', $tenantId, PDO::PARAM_INT);
        if($startDate !== null){
            $startDateStr = date('Y-m-d H:i:s', $startDate);
            $stmt->bindParam(':start_date', $startDateStr, PDO::PARAM_STR);
        }
        if($endDate !== null){
            $endDateStr = date('Y-m-d H:i:s', $endDate);
            $stmt->bindParam(':end_date', $endDateStr, PDO::PARAM_STR);
        }
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);

        if($orderType !== null){
            $stmt->bindParam(':order_type', $orderType, PDO::PARAM_STR);
        }
        if($orderStatus !== null){
            $stmt->bindParam(':order_status', $orderStatus, PDO::PARAM_STR);
        }
        if($paymentStatus !== null){
            $stmt->bindParam(':payment_status', $paymentStatus, PDO::PARAM_STR);
        }
        
        $stmt->execute();
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        if (!$rows) {
            return ['status' => 'failed', 'message' => 'No orders found'];
        }
        $displayTz = $inputData['tenantTz'] ?? 'Europe/London';
        foreach ($rows as &$r) {
            $r['total_order_modified_price'] = number_format((float)$r['total_order_modified_price'], 2, '.', '');
            if (!empty($r['created_at'])) {
                $rawUtc              = $r['created_at'];
                $r['created_at']     = utc_to_iso8601z($rawUtc);
                $r['created_at_local'] = format_utc_for_tz($rawUtc, $displayTz, 'Y-m-d H:i:s');
            } else {
                $r['created_at']      = null;
                $r['created_at_local'] = null;
            }
            $r = convertKeysToCamelCase($r);
        }
        unset($r);
        return ['status' => 'success', 'orders' => $rows, 'total' => ($countResult['total'] ?? 0)];
    }
?>