PRAGMA foreign_keys = ON;

-- Tenant table
CREATE TABLE tenant (
    id INTEGER PRIMARY KEY,
    parent_id INTEGER DEFAULT NULL,
    company_name TEXT NOT NULL,
    domain TEXT NOT NULL,
    company_number TEXT NOT NULL,
    food_hygiene_id TEXT,
    primary_contact_name TEXT NOT NULL,
    tenant_status TEXT DEFAULT 'pending',
    payment_provider TEXT DEFAULT 'stripe',
    payment_api_key TEXT DEFAULT NULL,
    minimum_order_amount DECIMAL(10, 2) DEFAULT 0.00,
    small_order_fee DECIMAL(10, 2) DEFAULT 0.00,
    is_deleted INTEGER DEFAULT 0,
    max_distance DECIMAL(10, 2) DEFAULT 0.00,
    active_delivery INTEGER(1) DEFAULT 1,
    active_pickup INTEGER(1) DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (parent_id) REFERENCES tenant(id)
);


-- Tenant Contact
CREATE TABLE tenant_contact (
    id INTEGER PRIMARY KEY,
    tenant_id INTEGER NOT NULL,
    user_id INTEGER NULL,
    contact_type TEXT NOT NULL,  -- mobile, email, etc.
    is_primary TEXT DEFAULT 'no', -- yes/no
    label TEXT NOT NULL,
    contact_value TEXT NOT NULL,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (tenant_id) REFERENCES tenant(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES user_login(pk_user_id) ON DELETE SET NULL
);

-- Tenant Theme
CREATE TABLE tenant_theme (
    id INTEGER PRIMARY KEY,
    tenant_id INTEGER NOT NULL,
    theme_name TEXT NOT NULL,
    primary_color TEXT NOT NULL,
    secondary_color TEXT NOT NULL,
    text_color TEXT NOT NULL,
    secondary_text_color TEXT NOT NULL,
    accent_color TEXT NOT NULL,
    background_color TEXT NOT NULL,
    button_color TEXT NOT NULL,
    theme_status TEXT DEFAULT 'draft', -- active, draft, archived
    is_applied TEXT DEFAULT 'no',     -- yes/no
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (tenant_id) REFERENCES tenant(id) ON DELETE CASCADE
);

-- Tenant Business Hours
CREATE TABLE tenant_business_hours (
    id INTEGER PRIMARY KEY,
    tenant_id INTEGER NOT NULL,
    hours_category TEXT NOT NULL,    -- store, delivery, etc.
    opening_day TEXT DEFAULT NULL,       -- Monday, Tuesday, etc.
    specific_date DATE DEFAULT NULL,
    open_time TIME DEFAULT NULL,
    close_time TIME DEFAULT NULL,
    is_closed INTEGER DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (tenant_id) REFERENCES tenant(id) ON DELETE CASCADE
);

-- Tenant history
CREATE TABLE tenant_history (
    id INTEGER PRIMARY KEY,
    tenant_id INTEGER NOT NULL,
    action_type TEXT NOT NULL,      -- created, updated, deleted
    table_of_source TEXT NOT NULL,  -- tenant, tenant_contact, etc.
    old_value TEXT DEFAULT NULL,    -- JSON string of old values
    action_by INTEGER NOT NULL,        -- user_id or system
    action_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE tenant_delivery_rules (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    tenant_id INTEGER NOT NULL,
    minimum_delivery_amount REAL NOT NULL,
    delivery_charge REAL NOT NULL,
    is_active INTEGER NOT NULL DEFAULT 0,
    is_deleted INTEGER NOT NULL DEFAULT 0,
    created_at TEXT DEFAULT (datetime('now')),
    updated_at TEXT DEFAULT (datetime('now'))
);

CREATE TABLE tenant_admin_fees (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    tenant_id INTEGER NOT NULL,
    customer_admin_fee REAL NOT NULL DEFAULT 0.00,
    tenant_admin_fee   REAL NOT NULL DEFAULT 0.00,
    is_active          INTEGER NOT NULL DEFAULT 0,
    is_deleted         INTEGER NOT NULL DEFAULT 0,
    created_at TEXT DEFAULT (datetime('now')),
    updated_at TEXT DEFAULT (datetime('now'))
);


CREATE VIEW tenant_upcoming_hours AS 
WITH RECURSIVE days(seq, generated_date, day_name) AS (
    SELECT
        0,
        DATE('now'),
        CASE strftime('%w', DATE('now'))
            WHEN '0' THEN 'Sunday'
            WHEN '1' THEN 'Monday'
            WHEN '2' THEN 'Tuesday'
            WHEN '3' THEN 'Wednesday'
            WHEN '4' THEN 'Thursday'
            WHEN '5' THEN 'Friday'
            WHEN '6' THEN 'Saturday'
        END
    UNION ALL
    SELECT
        seq + 1,
        DATE('now', '+' || (seq + 1) || ' day'),
        CASE strftime('%w', DATE('now', '+' || (seq + 1) || ' day'))
            WHEN '0' THEN 'Sunday'
            WHEN '1' THEN 'Monday'
            WHEN '2' THEN 'Tuesday'
            WHEN '3' THEN 'Wednesday'
            WHEN '4' THEN 'Thursday'
            WHEN '5' THEN 'Friday'
            WHEN '6' THEN 'Saturday'
        END
    FROM days
    WHERE seq < 13
)
SELECT
    d.generated_date,
    bh.tenant_id,
    bh.hours_category,
    bh.opening_day,
    CASE WHEN bh.is_closed = 1 THEN NULL ELSE bh.open_time END AS open_time,
    CASE WHEN bh.is_closed = 1 THEN NULL ELSE bh.close_time END AS close_time,
    bh.is_closed
FROM days d
JOIN tenant_business_hours bh
    ON (
        bh.specific_date = d.generated_date
        AND bh.is_deleted = 0
    )
    OR (
        bh.specific_date IS NULL
        AND bh.opening_day = d.day_name
        AND bh.is_deleted = 0
        AND NOT EXISTS (
            SELECT 1
            FROM tenant_business_hours sp
            WHERE sp.tenant_id = bh.tenant_id
              AND sp.specific_date = d.generated_date
              AND sp.is_deleted = 0
        )
    )
ORDER BY bh.tenant_id, d.generated_date, bh.hours_category;
