-- Cascading delete script for login/user cleanup
-- This script deletes user records only if they are exclusively connected to the tenant being deleted
-- Working from dependent tables to maintain referential integrity
-- 
-- Usage: Replace @tenant_id with the actual tenant ID to delete
-- Example: SET @tenant_id = 123;

SET @tenant_id = :tenant_id; -- Replace with actual tenant ID

-- Step 1: Create temporary table to identify users exclusively connected to this tenant
CREATE TEMPORARY TABLE temp_users_to_delete AS
SELECT DISTINCT up.fk_user_id
FROM user_permissions up
INNER JOIN user_system us ON up.fk_system_id = us.pk_system_id
WHERE us.fk_tenant_id = @tenant_id
  AND up.fk_user_id NOT IN (
    -- Exclude users who have permissions in other tenants
    SELECT DISTINCT up2.fk_user_id
    FROM user_permissions up2
    INNER JOIN user_system us2 ON up2.fk_system_id = us2.pk_system_id
    WHERE us2.fk_tenant_id != @tenant_id
  );

-- Step 2: Delete login logs for users who will be deleted
DELETE FROM user_login_log 
WHERE user_id IN (SELECT fk_user_id FROM temp_users_to_delete);

-- Step 3: Delete user permissions for this tenant (all users, not just those being deleted)
DELETE up FROM user_permissions up
INNER JOIN user_system us ON up.fk_system_id = us.pk_system_id
WHERE us.fk_tenant_id = @tenant_id;

-- Step 4: Delete user system records for this tenant
DELETE FROM user_system 
WHERE fk_tenant_id = @tenant_id;

-- Step 5: Delete users who are exclusively connected to this tenant
DELETE FROM user_login 
WHERE pk_user_id IN (SELECT fk_user_id FROM temp_users_to_delete);

-- Clean up temporary table
DROP TEMPORARY TABLE temp_users_to_delete;

-- Verification queries (optional - uncomment to verify deletion)
/*
-- Check that no user_system records remain for this tenant
SELECT 'User systems remaining for tenant' as check_type, COUNT(*) as count 
FROM user_system WHERE fk_tenant_id = @tenant_id
UNION ALL
-- Check that no permissions remain for this tenant
SELECT 'Permissions remaining for tenant' as check_type, COUNT(*) as count 
FROM user_permissions up 
INNER JOIN user_system us ON up.fk_system_id = us.pk_system_id 
WHERE us.fk_tenant_id = @tenant_id
UNION ALL
-- Show remaining users (these should be users connected to other tenants)
SELECT 'Total users remaining in system' as check_type, COUNT(*) as count 
FROM user_login
UNION ALL
-- Show users who still have permissions in other tenants
SELECT 'Users with permissions in other tenants' as check_type, COUNT(DISTINCT up.fk_user_id) as count 
FROM user_permissions up 
INNER JOIN user_system us ON up.fk_system_id = us.pk_system_id 
WHERE us.fk_tenant_id != @tenant_id;
*/