-- Full menu schema and audit triggers (with DELIMITER $$ wrapping)


-- Menu Table
CREATE TABLE menu_menus (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    tenant_id INTEGER NOT NULL,
    menu_name TEXT NOT NULL,
    menu_description TEXT,
    is_active INTEGER DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

-- Categories
CREATE TABLE menu_categories (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    menu_id INTEGER NOT NULL,
    parent_id INTEGER,
    category_name TEXT NOT NULL,
    category_description TEXT,
    is_deleted INTEGER DEFAULT 0,
    is_active INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (menu_id) REFERENCES menu_menus(id),
    FOREIGN KEY (parent_id) REFERENCES menu_categories(id)
);

-- Items
CREATE TABLE menu_items (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    item_name TEXT NOT NULL,
    item_description TEXT,
    base_price REAL,
    nutrition_id INTEGER,
    is_deleted INTEGER DEFAULT 0,
    is_active INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (nutrition_id) REFERENCES menu_nutrition_profiles(id)
);

-- Category-Item Linking
CREATE TABLE menu_category_items (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    category_id INTEGER NOT NULL,
    item_id INTEGER NOT NULL,
    is_deleted INTEGER DEFAULT 0,
    is_active INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES menu_categories(id),
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
);

-- Nutrition Profiles
CREATE TABLE menu_nutrition_profiles (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    calories INTEGER DEFAULT 0,
    protein_grams REAL DEFAULT 0,
    fat_grams REAL DEFAULT 0,
    carbs_grams REAL DEFAULT 0,
    sugar_grams REAL DEFAULT 0,
    salt_grams REAL DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

-- Option Groups
CREATE TABLE menu_option_groups (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    item_id INTEGER NOT NULL,
    option_group_name TEXT NOT NULL,
    option_group_description TEXT,
    min_select INTEGER DEFAULT 0,
    max_select INTEGER DEFAULT 1,
    is_required INTEGER DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
);

-- Option Items
CREATE TABLE menu_option_items (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    group_id INTEGER NOT NULL,
    option_item_name TEXT NOT NULL,
    option_item_description TEXT,
    price_adjustment REAL DEFAULT 0,
    nutrition_id INTEGER,
    is_default INTEGER DEFAULT 0,
    allow_toggle_only INTEGER DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (group_id) REFERENCES menu_option_groups(id),
    FOREIGN KEY (nutrition_id) REFERENCES menu_nutrition_profiles(id)
);

-- Meals
CREATE TABLE menu_meals (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    category_id INTEGER NOT NULL,
    meal_name TEXT NOT NULL,
    meal_description TEXT,
    base_price REAL,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES menu_categories(id)
);

-- Meal Parts
CREATE TABLE menu_meal_parts (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    meal_id INTEGER NOT NULL,
    meal_part_name TEXT NOT NULL,
    meal_part_description TEXT,
    min_select INTEGER DEFAULT 1,
    max_select INTEGER DEFAULT 1,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (meal_id) REFERENCES menu_meals(id)
);

-- Meal Part Items
CREATE TABLE menu_meal_part_items (
    part_id INTEGER NOT NULL,
    item_id INTEGER NOT NULL,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (part_id, item_id),
    FOREIGN KEY (part_id) REFERENCES menu_meal_parts(id),
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
);

-- =====================
-- AUDIT LOG TABLE SETUP
-- =====================
CREATE TABLE IF NOT EXISTS menu_audit_log (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    tenant_id INTEGER NOT NULL,
    table_of_source TEXT NOT NULL,
    record_id INTEGER NULL,
    action_type TEXT NOT NULL,
    old_value TEXT,
    action_by INTEGER DEFAULT 0,
    action_at DATETIME DEFAULT CURRENT_TIMESTAMP
)

DELIMITER $$


CREATE TRIGGER trg_only_one_active_menu
BEFORE INSERT ON menu_menus
FOR EACH ROW
WHEN NEW.is_active = 1
BEGIN
    UPDATE menu_menus
    SET is_active = 0
    WHERE tenant_id = NEW.tenant_id AND is_active = 1 AND is_deleted = 0;
END $$

CREATE TRIGGER trg_only_one_active_menu_update
BEFORE UPDATE ON menu_menus
FOR EACH ROW
WHEN NEW.is_active = 1 AND OLD.is_active != 1
BEGIN
    UPDATE menu_menus
    SET is_active = 0
    WHERE tenant_id = NEW.tenant_id AND id != NEW.id AND is_active = 1 AND is_deleted = 0;
END $$


-- =============================
-- menu_items: UPDATE & DELETE
-- =============================
CREATE TRIGGER trg_menu_items_update
AFTER UPDATE ON menu_items
FOR EACH ROW
BEGIN
    DECLARE changes TEXT DEFAULT '';
    DECLARE tenantId INTEGER;

    SELECT m.tenant_id INTO tenantId
    FROM menu_category_items ci
    JOIN menu_categories c ON ci.category_id = c.id
    JOIN menu_menus m ON c.menu_id = m.id
    WHERE ci.item_id = NEW.id
    LIMIT 1;

    IF OLD.item_name != NEW.item_name THEN
        SET changes = changes || 'item_name: from "' || OLD.item_name || '" to "' || NEW.item_name || '"; ';
    END IF;
    IF OLD.base_price != NEW.base_price THEN
        SET changes = changes || 'base_price: from ' || OLD.base_price || ' to ' || NEW.base_price || '; ';
    END IF;
    IF OLD.is_deleted != NEW.is_deleted THEN
        SET changes = changes || 'is_deleted: from ' || OLD.is_deleted || ' to ' || NEW.is_deleted || '; ';
    END IF;

    IF LENGTH(changes) > 0 THEN
        INSERT INTO menu_audit_log (
            tenant_id, table_of_source, record_id, action_type, old_value, action_by
        ) VALUES (
            tenantId, 'menu_items', NEW.id, 'updated', changes, 0
        );
    END IF;
END$$

CREATE TRIGGER trg_menu_items_delete
BEFORE DELETE ON menu_items
FOR EACH ROW
BEGIN
    DECLARE tenantId INTEGER;

    SELECT m.tenant_id INTO tenantId
    FROM menu_category_items ci
    JOIN menu_categories c ON ci.category_id = c.id
    JOIN menu_menus m ON c.menu_id = m.id
    WHERE ci.item_id = OLD.id
    LIMIT 1;

    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    ) VALUES (
        tenantId, 'menu_items', OLD.id, 'deleted',
        'item_name=' || OLD.item_name || '; base_price=' || OLD.base_price || '; is_deleted=' || OLD.is_deleted,
        0
    );
END$$

-- =============================
-- menu_menus: UPDATE & DELETE
-- =============================
CREATE TRIGGER trg_menu_menus_update
AFTER UPDATE ON menu_menus
FOR EACH ROW
BEGIN
    DECLARE changes TEXT DEFAULT '';

    IF OLD.menu_name != NEW.menu_name THEN
        SET changes = changes || 'menu_name: from "' || OLD.menu_name || '" to "' || NEW.menu_name || '"; ';
    END IF;
    IF OLD.is_active != NEW.is_active THEN
        SET changes = changes || 'is_active: from ' || OLD.is_active || ' to ' || NEW.is_active || '; ';
    END IF;
    IF OLD.is_deleted != NEW.is_deleted THEN
        SET changes = changes || 'is_deleted: from ' || OLD.is_deleted || ' to ' || NEW.is_deleted || '; ';
    END IF;

    IF LENGTH(changes) > 0 THEN
        INSERT INTO menu_audit_log (
            tenant_id, table_of_source, record_id, action_type, old_value, action_by
        ) VALUES (
            NEW.tenant_id, 'menu_menus', NEW.id, 'updated', changes, 0
        );
    END IF;
END$$

CREATE TRIGGER trg_menu_menus_delete
BEFORE DELETE ON menu_menus
FOR EACH ROW
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    ) VALUES (
        OLD.tenant_id, 'menu_menus', OLD.id, 'deleted',
        'menu_name=' || OLD.menu_name || '; is_active=' || OLD.is_active || '; is_deleted=' || OLD.is_deleted,
        0
    );
END$$

-- =============================
-- menu_categories: UPDATE & DELETE
-- =============================
CREATE TRIGGER trg_menu_categories_update
AFTER UPDATE ON menu_categories
FOR EACH ROW
BEGIN
    DECLARE changes TEXT DEFAULT '';

    IF OLD.category_name != NEW.category_name THEN
        SET changes = changes || 'category_name: from "' || OLD.category_name || '" to "' || NEW.category_name || '"; ';
    END IF;
    IF OLD.is_deleted != NEW.is_deleted THEN
        SET changes = changes || 'is_deleted: from ' || OLD.is_deleted || ' to ' || NEW.is_deleted || '; ';
    END IF;

    IF LENGTH(changes) > 0 THEN
        INSERT INTO menu_audit_log (
            tenant_id, table_of_source, record_id, action_type, old_value, action_by
        ) VALUES (
            (SELECT tenant_id FROM menu_menus WHERE id = NEW.menu_id),
            'menu_categories', NEW.id, 'updated', changes, 0
        );
    END IF;
END$$

CREATE TRIGGER trg_menu_categories_delete
BEFORE DELETE ON menu_categories
FOR EACH ROW
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    ) VALUES (
        (SELECT tenant_id FROM menu_menus WHERE id = OLD.menu_id),
        'menu_categories', OLD.id, 'deleted',
        'category_name=' || OLD.category_name || '; is_deleted=' || OLD.is_deleted,
        0
    );
END$$

-- =============================
-- menu_meals: UPDATE & DELETE
-- =============================
CREATE TRIGGER trg_menu_meals_update
AFTER UPDATE ON menu_meals
FOR EACH ROW
BEGIN
    DECLARE changes TEXT DEFAULT '';

    IF OLD.meal_name != NEW.meal_name THEN
        SET changes = changes || 'meal_name: from "' || OLD.meal_name || '" to "' || NEW.meal_name || '"; ';
    END IF;
    IF OLD.base_price != NEW.base_price THEN
        SET changes = changes || 'base_price: from ' || OLD.base_price || ' to ' || NEW.base_price || '; ';
    END IF;
    IF OLD.is_deleted != NEW.is_deleted THEN
        SET changes = changes || 'is_deleted: from ' || OLD.is_deleted || ' to ' || NEW.is_deleted || '; ';
    END IF;

    IF LENGTH(changes) > 0 THEN
        INSERT INTO menu_audit_log (
            tenant_id, table_of_source, record_id, action_type, old_value, action_by
        ) VALUES (
            (SELECT tenant_id FROM menu_menus WHERE id = (SELECT menu_id FROM menu_categories WHERE id = NEW.category_id)),
            'menu_meals', NEW.id, 'updated', changes, 0
        );
    END IF;
END$$

CREATE TRIGGER trg_menu_meals_delete
BEFORE DELETE ON menu_meals
FOR EACH ROW
BEGIN
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    ) VALUES (
        (SELECT tenant_id FROM menu_menus WHERE id = (SELECT menu_id FROM menu_categories WHERE id = OLD.category_id)),
        'menu_meals', OLD.id, 'deleted',
        'meal_name=' || OLD.meal_name || '; base_price=' || OLD.base_price || '; is_deleted=' || OLD.is_deleted,
        0
    );
END$$

-- =============================
-- menu_option_items: UPDATE & DELETE
-- =============================
CREATE TRIGGER trg_menu_option_items_update
AFTER UPDATE ON menu_option_items
FOR EACH ROW
BEGIN
    DECLARE changes TEXT DEFAULT '';
    DECLARE tenantId INTEGER;

    SELECT m.tenant_id INTO tenantId
    FROM menu_option_groups g
    JOIN menu_items i ON g.item_id = i.id
    JOIN menu_category_items ci ON ci.item_id = i.id
    JOIN menu_categories c ON ci.category_id = c.id
    JOIN menu_menus m ON c.menu_id = m.id
    WHERE g.id = NEW.group_id
    LIMIT 1;

    IF OLD.option_item_name != NEW.option_item_name THEN
        SET changes = changes || 'option_item_name: from "' || OLD.option_item_name || '" to "' || NEW.option_item_name || '"; ';
    END IF;
    IF OLD.price_adjustment != NEW.price_adjustment THEN
        SET changes = changes || 'price_adjustment: from ' || OLD.price_adjustment || ' to ' || NEW.price_adjustment || '; ';
    END IF;
    IF OLD.is_deleted != NEW.is_deleted THEN
        SET changes = changes || 'is_deleted: from ' || OLD.is_deleted || ' to ' || NEW.is_deleted || '; ';
    END IF;

    IF LENGTH(changes) > 0 THEN
        INSERT INTO menu_audit_log (
            tenant_id, table_of_source, record_id, action_type, old_value, action_by
        ) VALUES (
            tenantId, 'menu_option_items', NEW.id, 'updated', changes, 0
        );
    END IF;
END$$

CREATE TRIGGER trg_menu_option_items_delete
BEFORE DELETE ON menu_option_items
FOR EACH ROW
BEGIN
    DECLARE tenantId INTEGER;

    SELECT m.tenant_id INTO tenantId
    FROM menu_option_groups g
    JOIN menu_items i ON g.item_id = i.id
    JOIN menu_category_items ci ON ci.item_id = i.id
    JOIN menu_categories c ON ci.category_id = c.id
    JOIN menu_menus m ON c.menu_id = m.id
    WHERE g.id = OLD.group_id
    LIMIT 1;

    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    ) VALUES (
        tenantId, 'menu_option_items', OLD.id, 'deleted',
        'option_item_name=' || OLD.option_item_name || '; price_adjustment=' || OLD.price_adjustment || '; is_deleted=' || OLD.is_deleted,
        0
    );
END$$

CREATE TRIGGER trg_menu_menus_is_deleted
AFTER UPDATE ON menu_menus
FOR EACH ROW
WHEN NEW.is_deleted != OLD.is_deleted
BEGIN
    UPDATE menu_categories
    SET is_deleted = NEW.is_deleted
    WHERE menu_id = NEW.id;
END$$

CREATE TRIGGER trg_mark_category_items_deleted
AFTER UPDATE OF is_deleted ON menu_items
FOR EACH ROW
WHEN NEW.is_deleted = 1 AND OLD.is_deleted = 0
BEGIN
    UPDATE menu_category_items
    SET is_deleted = 1,
        updated_at = CURRENT_TIMESTAMP
    WHERE item_id = NEW.id;
END$$

DELIMITER ;
