-- Full menu schema and audit triggers (with DELIMITER $$ wrapping)


-- Menu Table
CREATE TABLE menu_menus (
    id INTEGER PRIMARY KEY AUTO_INCREMENT,
    tenant_id INTEGER NOT NULL,
    menu_name TEXT NOT NULL,
    menu_description TEXT,
    is_active INTEGER DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

-- Categories
CREATE TABLE menu_categories (
    id INTEGER PRIMARY KEY AUTO_INCREMENT,
    menu_id INTEGER NOT NULL,
    parent_id INTEGER,
    category_name TEXT NOT NULL,
    category_description TEXT,
    is_deleted INTEGER DEFAULT 0,
    is_active INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (menu_id) REFERENCES menu_menus(id),
    FOREIGN KEY (parent_id) REFERENCES menu_categories(id)
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

-- Nutrition Profiles
CREATE TABLE menu_nutrition_profiles (
    id INTEGER PRIMARY KEY AUTO_INCREMENT,
    calories INTEGER DEFAULT 0,
    protein_grams REAL DEFAULT 0,
    fat_grams REAL DEFAULT 0,
    carbs_grams REAL DEFAULT 0,
    sugar_grams REAL DEFAULT 0,
    salt_grams REAL DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

-- Items
CREATE TABLE menu_items (
    id INTEGER PRIMARY KEY AUTO_INCREMENT,
    item_name TEXT NOT NULL,
    item_description TEXT,
    base_price REAL,
    nutrition_id INTEGER,
    is_deleted INTEGER DEFAULT 0,
    is_active INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (nutrition_id) REFERENCES menu_nutrition_profiles(id)
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

-- Category-Item Linking
CREATE TABLE menu_category_items (
    id INTEGER PRIMARY KEY AUTO_INCREMENT,
    category_id INTEGER NOT NULL,
    item_id INTEGER NOT NULL,
    is_deleted INTEGER DEFAULT 0,
    is_active INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;



-- Option Groups
CREATE TABLE menu_option_groups (
    id INTEGER PRIMARY KEY AUTO_INCREMENT,
    item_id INTEGER NOT NULL,
    group_name TEXT NOT NULL,
    group_description TEXT,
    min_select INTEGER DEFAULT 0,
    max_select INTEGER DEFAULT 1,
    is_required INTEGER DEFAULT 0,
    is_deleted INTEGER DEFAULT 0,
    is_active INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

-- Option Items
CREATE TABLE menu_option_extra (
    id                  INTEGER PRIMARY KEY AUTO_INCREMENT,
    tenant_id           INTEGER NOT NULL,
    extra_name          TEXT NOT NULL,
    extra_description   TEXT,
    price_adjustment    REAL DEFAULT 0,
    nutrition_id        INTEGER NULL,
    is_deleted          INTEGER DEFAULT 0,
    is_active           INTEGER DEFAULT 0,
    created_at          DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at          DATETIME DEFAULT CURRENT_TIMESTAMP
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;
--    FOREIGN KEY (nutrition_id) REFERENCES menu_nutrition_profiles(id)

create table menu_option_joins (
    id                  INTEGER PRIMARY KEY AUTO_INCREMENT,
    group_id            INTEGER NOT NULL,
    extra_id            INTEGER NOT NULL,
    price_adjustment    REAL DEFAULT 0,
    is_default          INTEGER DEFAULT 0,
    is_active           INTEGER DEFAULT 0,
    is_deleted          INTEGER DEFAULT 0,
    created_at          DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at          DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (group_id) REFERENCES menu_option_groups(id),
    FOREIGN KEY (extra_id) REFERENCES menu_option_extra(id)
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

-- Meals
CREATE TABLE menu_meals (
    id INTEGER PRIMARY KEY AUTO_INCREMENT,
    category_id INTEGER NOT NULL,
    meal_name TEXT NOT NULL,
    meal_description TEXT,
    base_price REAL,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES menu_categories(id)
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

-- Meal Parts
CREATE TABLE menu_meal_parts (
    id INTEGER PRIMARY KEY AUTO_INCREMENT,
    meal_id INTEGER NOT NULL,
    meal_part_name TEXT NOT NULL,
    meal_part_description TEXT,
    min_select INTEGER DEFAULT 1,
    max_select INTEGER DEFAULT 1,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (meal_id) REFERENCES menu_meals(id)
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

-- Meal Part Items
CREATE TABLE menu_meal_part_items (
    part_id INTEGER NOT NULL,
    item_id INTEGER NOT NULL,
    is_deleted INTEGER DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    PRIMARY KEY (part_id, item_id),
    FOREIGN KEY (part_id) REFERENCES menu_meal_parts(id),
    FOREIGN KEY (item_id) REFERENCES menu_items(id)
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

-- =====================
-- AUDIT LOG TABLE SETUP
-- =====================
CREATE TABLE IF NOT EXISTS menu_audit_log (
    id INTEGER PRIMARY KEY AUTO_INCREMENT,
    tenant_id INTEGER NULL DEFAULT NULL,
    table_of_source TEXT NOT NULL,
    record_id INTEGER NULL,
    action_type TEXT NOT NULL,
    old_value TEXT,
    action_by INTEGER DEFAULT 0,
    action_at DATETIME DEFAULT CURRENT_TIMESTAMP
)ENGINE=InnoDB DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

DELIMITER $$
/*
CREATE TRIGGER trg_only_one_active_menu
BEFORE INSERT ON menu_menus
FOR EACH ROW
BEGIN
    IF NEW.is_active = 1 THEN
        UPDATE menu_menus
        SET is_active = 0
        WHERE tenant_id = NEW.tenant_id
          AND is_active = 1
          AND is_deleted = 0;
    END IF;
END$$

DELIMITER ;


    DELIMITER $$

CREATE TRIGGER trg_only_one_active_menu_update
BEFORE UPDATE ON menu_menus
FOR EACH ROW
BEGIN
    IF NEW.is_active = 1 AND OLD.is_active != 1 THEN
        UPDATE menu_menus
        SET is_active = 0
        WHERE tenant_id = NEW.tenant_id
          AND id != NEW.id
          AND is_active = 1
          AND is_deleted = 0;
    END IF;
END$$

DELIMITER ;
*/


DROP TRIGGER IF EXISTS trg_menu_items_update;
DELIMITER $$
CREATE TRIGGER trg_menu_items_update
AFTER UPDATE ON menu_items
FOR EACH ROW
BEGIN
    DECLARE changes TEXT DEFAULT '';
    DECLARE tenantId INT;

    -- First, retrieve tenant ID
    SELECT m.tenant_id INTO tenantId
    FROM menu_category_items ci
    JOIN menu_categories c ON ci.category_id = c.id
    JOIN menu_menus m ON c.menu_id = m.id
    WHERE ci.item_id = NEW.id
    LIMIT 1;

    -- Append changes using CONCAT
    IF OLD.item_name != NEW.item_name THEN
        SET changes = CONCAT(changes, 'item_name: from "', OLD.item_name, '" to "', NEW.item_name, '"; ');
    END IF;
    IF OLD.base_price != NEW.base_price THEN
        SET changes = CONCAT(changes, 'base_price: from ', OLD.base_price, ' to ', NEW.base_price, '; ');
    END IF;
    IF OLD.is_deleted != NEW.is_deleted THEN
        SET changes = CONCAT(changes, 'is_deleted: from ', OLD.is_deleted, ' to ', NEW.is_deleted, '; ');
    END IF;

    -- Insert into audit log if anything changed
    IF LENGTH(changes) > 0 THEN
        INSERT INTO menu_audit_log (
            tenant_id, table_of_source, record_id, action_type, old_value, action_by
        ) VALUES (
            tenantId, 'menu_items', NEW.id, 'updated', changes, 0
        );
    END IF;
END$$

DELIMITER ;


DROP TRIGGER IF EXISTS trg_menu_items_delete;
DELIMITER $$
CREATE TRIGGER trg_menu_items_delete
BEFORE DELETE ON menu_items
FOR EACH ROW
BEGIN
    DECLARE tenantId INT;
    DECLARE oldValue TEXT;

    -- Get tenant ID
    SELECT m.tenant_id INTO tenantId
    FROM menu_category_items ci
    JOIN menu_categories c ON ci.category_id = c.id
    JOIN menu_menus m ON c.menu_id = m.id
    WHERE ci.item_id = OLD.id
    LIMIT 1;

    -- Construct old value string
    SET oldValue = CONCAT(
        'item_name=', OLD.item_name, '; ',
        'base_price=', OLD.base_price, '; ',
        'is_deleted=', OLD.is_deleted
    );

    -- Insert into audit log
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    ) VALUES (
        tenantId, 'menu_items', OLD.id, 'deleted', oldValue, 0
    );
END$$

DELIMITER ;

DROP TRIGGER IF EXISTS trg_menu_menus_update;
DELIMITER $$
CREATE TRIGGER trg_menu_menus_update
AFTER UPDATE ON menu_menus
FOR EACH ROW
BEGIN
    DECLARE changes TEXT DEFAULT '';

    IF OLD.menu_name != NEW.menu_name THEN
        SET changes = CONCAT(changes, 'menu_name: from "', OLD.menu_name, '" to "', NEW.menu_name, '"; ');
    END IF;

    IF OLD.is_active != NEW.is_active THEN
        SET changes = CONCAT(changes, 'is_active: from ', OLD.is_active, ' to ', NEW.is_active, '; ');
    END IF;

    IF OLD.is_deleted != NEW.is_deleted THEN
        SET changes = CONCAT(changes, 'is_deleted: from ', OLD.is_deleted, ' to ', NEW.is_deleted, '; ');
    END IF;

    IF LENGTH(changes) > 0 THEN
        INSERT INTO menu_audit_log (
            tenant_id, table_of_source, record_id, action_type, old_value, action_by
        ) VALUES (
            NEW.tenant_id, 'menu_menus', NEW.id, 'updated', changes, 0
        );
    END IF;
END$$

DELIMITER ;

DROP TRIGGER IF EXISTS trg_menu_menus_delete;
DELIMITER $$
CREATE TRIGGER trg_menu_menus_delete
BEFORE DELETE ON menu_menus
FOR EACH ROW
BEGIN
    DECLARE oldValue TEXT;

    SET oldValue = CONCAT(
        'menu_name=', OLD.menu_name, '; ',
        'is_active=', OLD.is_active, '; ',
        'is_deleted=', OLD.is_deleted
    );

    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    ) VALUES (
        OLD.tenant_id, 'menu_menus', OLD.id, 'deleted', oldValue, 0
    );
END$$

DELIMITER ;

DROP TRIGGER IF EXISTS trg_menu_categories_update;
DELIMITER $$
CREATE TRIGGER trg_menu_categories_update
AFTER UPDATE ON menu_categories
FOR EACH ROW
BEGIN
    DECLARE changes TEXT DEFAULT '';
    DECLARE tenantId INT;

    -- Detect changes
    IF OLD.category_name != NEW.category_name THEN
        SET changes = CONCAT(changes, 'category_name: from "', OLD.category_name, '" to "', NEW.category_name, '"; ');
    END IF;

    IF OLD.is_deleted != NEW.is_deleted THEN
        SET changes = CONCAT(changes, 'is_deleted: from ', OLD.is_deleted, ' to ', NEW.is_deleted, '; ');
    END IF;

    -- If any changes detected, get tenant ID and insert into audit log
    IF LENGTH(changes) > 0 THEN
        SELECT m.tenant_id INTO tenantId
        FROM menu_menus m
        WHERE m.id = NEW.menu_id
        LIMIT 1;

        INSERT INTO menu_audit_log (
            tenant_id, table_of_source, record_id, action_type, old_value, action_by
        ) VALUES (
            tenantId, 'menu_categories', NEW.id, 'updated', changes, 0
        );
    END IF;
END$$

DELIMITER ;

DROP TRIGGER IF EXISTS trg_menu_categories_delete;
DELIMITER $$
CREATE TRIGGER trg_menu_categories_delete
BEFORE DELETE ON menu_categories
FOR EACH ROW
BEGIN
    DECLARE tenantId INT;
    DECLARE oldValue TEXT;

    -- Get tenant ID from the related menu
    SELECT tenant_id INTO tenantId
    FROM menu_menus
    WHERE id = OLD.menu_id
    LIMIT 1;

    -- Build the audit value string
    SET oldValue = CONCAT(
        'category_name=', OLD.category_name, '; ',
        'is_deleted=', OLD.is_deleted
    );

    -- Insert audit log entry
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    ) VALUES (
        tenantId, 'menu_categories', OLD.id, 'deleted', oldValue, 0
    );
END$$

DELIMITER ;

DROP TRIGGER IF EXISTS trg_menu_meals_update;
DELIMITER $$
CREATE TRIGGER trg_menu_meals_update
AFTER UPDATE ON menu_meals
FOR EACH ROW
BEGIN
    DECLARE changes TEXT DEFAULT '';
    DECLARE tenantId INT;

    -- Detect changes
    IF OLD.meal_name != NEW.meal_name THEN
        SET changes = CONCAT(changes, 'meal_name: from "', OLD.meal_name, '" to "', NEW.meal_name, '"; ');
    END IF;

    IF OLD.base_price != NEW.base_price THEN
        SET changes = CONCAT(changes, 'base_price: from ', OLD.base_price, ' to ', NEW.base_price, '; ');
    END IF;

    IF OLD.is_deleted != NEW.is_deleted THEN
        SET changes = CONCAT(changes, 'is_deleted: from ', OLD.is_deleted, ' to ', NEW.is_deleted, '; ');
    END IF;

    -- Lookup tenant ID if changes exist
    IF LENGTH(changes) > 0 THEN
        SELECT m.tenant_id INTO tenantId
        FROM menu_menus m
        JOIN menu_categories c ON c.menu_id = m.id
        WHERE c.id = NEW.category_id
        LIMIT 1;

        INSERT INTO menu_audit_log (
            tenant_id, table_of_source, record_id, action_type, old_value, action_by
        ) VALUES (
            tenantId, 'menu_meals', NEW.id, 'updated', changes, 0
        );
    END IF;
END$$

DELIMITER ;

DROP TRIGGER IF EXISTS trg_menu_meals_delete;
DELIMITER $$
CREATE TRIGGER trg_menu_meals_delete
BEFORE DELETE ON menu_meals
FOR EACH ROW
BEGIN
    DECLARE tenantId INT;
    DECLARE oldValue TEXT;

    -- Get tenant ID via join
    SELECT m.tenant_id INTO tenantId
    FROM menu_menus m
    JOIN menu_categories c ON c.menu_id = m.id
    WHERE c.id = OLD.category_id
    LIMIT 1;

    -- Build the oldValue string
    SET oldValue = CONCAT(
        'meal_name=', OLD.meal_name, '; ',
        'base_price=', OLD.base_price, '; ',
        'is_deleted=', OLD.is_deleted
    );

    -- Insert audit entry
    INSERT INTO menu_audit_log (
        tenant_id, table_of_source, record_id, action_type, old_value, action_by
    ) VALUES (
        tenantId, 'menu_meals', OLD.id, 'deleted', oldValue, 0
    );
END$$

DELIMITER ;

DROP TRIGGER IF EXISTS trg_menu_menus_is_deleted;
DELIMITER $$
CREATE TRIGGER trg_menu_menus_is_deleted
AFTER UPDATE ON menu_menus
FOR EACH ROW
BEGIN
    IF NEW.is_deleted != OLD.is_deleted THEN
        UPDATE menu_categories
        SET is_deleted = NEW.is_deleted
        WHERE menu_id = NEW.id;
    END IF;
END$$

DELIMITER ;

DROP TRIGGER IF EXISTS trg_mark_category_items_deleted;
DELIMITER $$
CREATE TRIGGER trg_mark_category_items_deleted
AFTER UPDATE ON menu_items
FOR EACH ROW
BEGIN
    IF NEW.is_deleted = 1 AND OLD.is_deleted = 0 THEN
        UPDATE menu_category_items
        SET is_deleted = 1,
            updated_at = CURRENT_TIMESTAMP
        WHERE item_id = NEW.id;
    END IF;
END$$

DELIMITER ;



-- Trigger 1: Update joins when group is disabled or deleted
DROP TRIGGER IF EXISTS trg_option_group_disable_or_delete_joins;
DELIMITER $$
CREATE TRIGGER trg_option_group_disable_or_delete_joins
AFTER UPDATE ON menu_option_groups
FOR EACH ROW
BEGIN
    IF NEW.is_deleted = 1 OR NEW.is_active = 0 THEN
        UPDATE menu_option_joins
        SET 
            is_deleted = NEW.is_deleted,
            is_active = NEW.is_active,
            updated_at = CURRENT_TIMESTAMP
        WHERE group_id = NEW.id;
    END IF;
END$$
DELIMITER ;

DROP TRIGGER IF EXISTS trg_option_group_cascade_items;
DELIMITER $$
CREATE TRIGGER trg_option_group_cascade_items
AFTER UPDATE ON menu_option_groups
FOR EACH ROW
BEGIN
  IF NEW.is_deleted = 1 OR NEW.is_active = 0 THEN
    UPDATE menu_option_extra
    SET
      is_deleted = NEW.is_deleted,
      is_active = NEW.is_active,
      updated_at = CURRENT_TIMESTAMP
    WHERE id IN (
      SELECT extra_id
      FROM menu_extra_joins j1
      WHERE j1.group_id = NEW.id
        AND NOT EXISTS (
          SELECT 1 FROM menu_option_joins j2
          WHERE j2.extra_id = j1.extra_id
            AND j2.group_id != NEW.id
            AND j2.is_deleted = 0
        )
    );
  END IF;
END$$
DELIMITER ;

-- Trigger 3: Propagate deletion/disable from option items to joins
DROP TRIGGER IF EXISTS trg_option_extra_status_to_joins;
DELIMITER $$
CREATE TRIGGER trg_option_extra_status_to_joins
AFTER UPDATE ON menu_option_extra
FOR EACH ROW
BEGIN
    IF NEW.is_deleted = 1 OR NEW.is_active = 0 THEN
        UPDATE menu_option_joins
        SET 
            is_deleted = NEW.is_deleted,
            is_active = NEW.is_active,
            updated_at = CURRENT_TIMESTAMP
        WHERE extra_id = NEW.id;
    END IF;
END$$
DELIMITER ;
