<?php
    ini_set('error_log', 'php://stderr');
    error_log("Something went wrong");

    use PHPUnit\Framework\TestCase;
    
    require_once __DIR__ . '/../../core/unit-test/scripts/bootstrap.php'; // ✅ Use shared mock DB
    //require_once __DIR__ . '/../src/function.php';
    echo "\T13_MetaDataTest.php\n";
    
    // Print list of table that exist in the database
    //$tables = $GLOBALS['mockPdo']->query("SELECT name FROM sqlite_master WHERE type='table';")->fetchAll(PDO::FETCH_COLUMN);
    //echo("\n🔍 Tables: " . implode(', ', $tables). "\n");
    
    class T13_MetaDataTest extends TestCase
    {
        private $mockPdo;
        private static $lastModified;

    protected function setUp(): void{
        // ✅ Use the shared mock database connection
        $_SESSION = [];
        $_GET = [];
        $_POST = [];
        $_SERVER['REMOTE_ADDR'] = '127.0.0.1';
        $_SERVER['HTTP_REFERER'] = 'http://localhost/test';
        $this->mockPdo = $GLOBALS['mockPdo'];
        if (session_status() !== PHP_SESSION_ACTIVE) {
            session_start();
        }
    }

    private function executeFunctionPhp(): array {
        ob_start();
        require __DIR__ . '/../src/function.php';
        $output = ob_get_clean();
    
        // ✅ Ensure only valid JSON output
        $decodedOutput = json_decode($output, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Invalid JSON output: " . json_last_error_msg() . "\nOutput: " . $output);
        }
    
        return [$output, $decodedOutput];
    }

    public function testDummyTest(): void{
        $this->assertTrue(true);
    }

    public function test001ViewMetaDataAsSysAdmin(): void {
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 1;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'viewMetaDataSuperAdmin',
            'viewType' => 'superAdmin',
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('success', $decodedOutput['status'], "View MetaData as SysAdmin failed: " . $output);
        $this->assertIsArray($decodedOutput['data'], "Data is not an array.");
        $this->assertEquals(0, count($decodedOutput['data']), "No MetaData records found.");
    }
    public function test002ViewMetaDataPublic(): void {
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 4;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'viewMetaDataPublic',
            'viewType' => 'superAdmin',
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('success', $decodedOutput['status'], "View MetaData as SysAdmin failed: " . $output);
        $this->assertIsArray($decodedOutput['data'], "Data is not an array.");
        $this->assertEquals(0, count($decodedOutput['data']), "No MetaData records found.");
    }
    public function test003InsertMetaData(): void {
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 1;         // Role Value (4 = Owner)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'insertMetaData',
            'metaData' => [
                'metaKey' => 'test_key',
                'metaValue' => 'test_value',
                'isActive' => 1
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('success', $decodedOutput['status'], "Insert MetaData failed: " . $output);
        // Verify data in the database
        $stmt = $this->mockPdo->prepare("SELECT * FROM meta_data WHERE id = :id AND is_deleted = 0");
        $stmt->bindParam(':id', $decodedOutput['id']);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        //echo print_r($result, true); // or var_dump($result)
        $this->assertNotFalse($result, "No record found in database for inserted MetaData.");
        $this->assertEquals('test_value', $result['meta_value'], "Meta value does not match expected value.");
    }
    public function test004InsertMetaDataMissingRequiredField(): void {
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 1;         // Role Value (4 = Owner)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'insertMetaData',
            'metaData' => [
                // 'metaKey' is missing
                'metaValue' => 'test_value',
                'isActive' => 1
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('failed', $decodedOutput['status'], "Insert MetaData should have failed due to missing required field.");
        $this->assertStringContainsString('metaKey is required', $decodedOutput['message'], "Error message does not mention missing metaKey.");
    }
    public function test005InsertMetaDataOptionalFieldMissing(): void {
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 1;         // Role Value (4 = Owner)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'insertMetaData',
            'metaData' => [
                'metaKey' => 'test_key_optional',
                'metaValue' => 'shop_value',
                //'isActive' => 1 is missing, which is optional
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('success', $decodedOutput['status'], "Insert MetaData failed when optional field was missing: " . $output);
        // Verify data in the database
        $stmt = $this->mockPdo->prepare("SELECT * FROM meta_data WHERE id = :id AND is_deleted = 0");
        $stmt->bindParam(':id', $decodedOutput['id']);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        //echo print_r($result, true); // or var_dump($result)
        $this->assertNotFalse($result, "No record found in database for inserted MetaData with optional field missing.");
        $this->assertEquals('1', $result['is_active'], "isActive does not default to 1 when not provided.");
    }
    public function test006InsertMetaDataInvalidRoleType(): void {
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 4;         // Invalid Role Value
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'insertMetaData',
            'metaData' => [
                'metaKey' => 'test_key_invalid_role',
                'metaValue' => 'test_value',
                'isActive' => 1
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('failed', $decodedOutput['status'], "Insert MetaData should have failed due to invalid role type.");
        $this->assertStringContainsString('Access denied: insufficient permissions', $decodedOutput['message'], "Error message does not mention invalid action.");
    }
    public function test007ViewMetaDataAsSysAdminAfterInsert(): void {
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 1;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'viewMetaDataSuperAdmin',
            'viewType' => 'superAdmin',
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('success', $decodedOutput['status'], "View MetaData as SysAdmin failed after insert: " . $output);
        $this->assertIsArray($decodedOutput['data'], "Data is not an array.");
        $this->assertGreaterThanOrEqual(1, count($decodedOutput['data']), "No MetaData records found after insert.");
    }
    public function test008ViewMetaDataPublicAfterInsert(): void {
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 4;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'viewMetaDataPublic',
            'viewType' => 'public',
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('success', $decodedOutput['status'], "View MetaData as Public failed after insert: " . $output);
        $this->assertIsArray($decodedOutput['data'], "Data is not an array.");
        $this->assertGreaterThanOrEqual(1, count($decodedOutput['data']), "No MetaData records found after insert.");
    }
    public function test009ViewMetaDataAsSysAdminNoTenant(): void {
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 1;         // Role Value (4 = Owner)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = null;          // Tenant ID for the user
        $_POST = [
            'action' => 'viewMetaDataSuperAdmin',
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('failed', $decodedOutput['status'], "View MetaData as SysAdmin should have failed due to missing tenant ID.");
        $this->assertStringContainsString('Tenant ID not specified', $decodedOutput['message'], "Error message does not mention missing tenant ID.");
    }
    public function test010ViewMetaDataPublicNoTenant(): void {
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 4;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = null;          // Tenant ID for the user
        $_POST = [
            'action' => 'viewMetaDataPublic',
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('failed', $decodedOutput['status'], "View MetaData as Public should have failed due to missing tenant ID.");
        $this->assertStringContainsString('Tenant ID not specified', $decodedOutput['message'], "Error message does not mention missing tenant ID.");
    }
    public function test011UpdateMetaDataAsSysAdmin(): void {
        // sql query to view the data in meta_data table
        $stmt = $this->mockPdo->query("SELECT * FROM meta_data");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
        // Debug output
        //echo print_r($rows, true); // or var_dump($rows)
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 1;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'updateMetaData',
            'data' => [
                'id' => 1,
                'metaValue' => 'updated_value',
                'metaKey' => 'test_key'
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('success', $decodedOutput['status'], "Update MetaData as SysAdmin failed: " . $output);
        // Verify data in the database
        $stmt = $this->mockPdo->prepare("SELECT * FROM meta_data WHERE id = 1 AND is_deleted = 0");
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        //echo print_r($result, true); // or var_dump($result)
        $this->assertNotFalse($result, "No record found in database for updated MetaData.");
        $this->assertEquals('updated_value', $result['meta_value'], "Meta value was not updated correctly.");
        // View the history data 
        $stmt = $this->mockPdo->prepare("SELECT * FROM meta_data_history WHERE meta_data_id = 1");
        $stmt->execute();
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        //echo print_r($result, true); // or var_dump($result)
        $this->assertEquals('1', $result[0]['id']);
    }
    public function test012UpdateMetaDataInvalidRole(): void {
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 4;         // Role Value (4 = Owner)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'updateMetaData',
            'data' => [
                'id' => 1,
                'metaValue' => 'updated_value_invalid_role',
                'metaKey' => 'test_key'
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('failed', $decodedOutput['status'], "Update MetaData should have failed due to invalid role.");
        $this->assertStringContainsString('Access denied: insufficient permissions', $decodedOutput['message'], "Error message does not mention insufficient permissions.");
    }
    public function test013UpdateWithMissingId(): void{
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 1;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'updateMetaData',
            'data' => [
                'metaValue' => 'updated_value',
                'metaKey' => 'test_key'
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('failed', $decodedOutput['status'], "Update MetaData as Id is missing: " . $output);
        $this->assertEquals('Missing required Id field: id key: meta_data', $decodedOutput['message'], "The id should be missing");
    }
    public function test014UpdateWithMissingValues(): void{
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 1;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'updateMetaData',
            'data' => [
                'id' => 1
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('failed', $decodedOutput['status'], "Update MetaData as Id is missing: " . $output);
        $this->assertEquals('No fields to update other than ID', $decodedOutput['message'], "No Content to be updated");
    }
    public function test015SetMetaDataToDeActive(): void{
        $stmt = $this->mockPdo->query("SELECT is_active FROM meta_data where id = 1");
        $rows = $stmt->fetch(PDO::FETCH_ASSOC);
        // Debug output
        //echo print_r($rows, true); // or var_dump($rows)
        $isActive = $rows['is_active'];
        $this->assertEquals('1', $isActive, 'Active should be equal to 1');

        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 1;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'updateMetaData',
            'data' => [
                'id' => 1,
                'isActive' => 0
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('success', $decodedOutput['status'], "Update MetaData as Id is missing: " . $output);
        $this->assertEquals('Update completed successfully', $decodedOutput['message'], "ID 1 should be deactivated");

        $stmt = $this->mockPdo->query("SELECT is_active FROM meta_data where id = 1");
        $rows = $stmt->fetch(PDO::FETCH_ASSOC);
        // Debug output
        //echo print_r($rows, true); // or var_dump($rows)
        $isActive = $rows['is_active'];
        $this->assertEquals('0', $isActive, 'Active should be equal to 0');

        $this->mockPdo->query("UPDATE meta_data SET  is_active = 1  WHERE id = 1");
        $stmt = $this->mockPdo->query("SELECT is_active FROM meta_data      WHERE id = 1");
        $rows = $stmt->fetch(PDO::FETCH_ASSOC);
        $isActive = $rows['is_active'];
        $this->assertEquals('1', $isActive, 'Active should be equal to 1');

    }
    public function test016SetMetaDataNoAccess(): void{
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 6;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'updateMetaData',
            'data' => [
                'id' => 1,
                'isActive' => 0
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('failed', $decodedOutput['status'], "Update MetaData as Id is missing: " . $output);
        $this->assertEquals('Access denied: insufficient permissions', $decodedOutput['message'], "Access denied: insufficient permissions RoleValue 6 should not be able to preform action");
    }
    public function test017SetMetaDataToDeleted(): void{
        $stmt = $this->mockPdo->query("SELECT is_deleted FROM meta_data where id = 1");
        $rows = $stmt->fetch(PDO::FETCH_ASSOC);
        // Debug output
        //echo print_r($rows, true); // or var_dump($rows)
        $isDeleted = $rows['is_deleted'];
        $this->assertEquals('0', $isDeleted, 'Active should be equal to 1');

        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 1;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'deleteMetaData',
            'data' => [
                'id' => 1,
                'isDeleted' => 1
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('success', $decodedOutput['status'], "Update MetaData as Id is missing: " . $output);
        $this->assertEquals('Update completed successfully', $decodedOutput['message'], "ID 1 should be deactivated");

        $stmt = $this->mockPdo->query("SELECT is_deleted FROM meta_data where id = 1");
        $rows = $stmt->fetch(PDO::FETCH_ASSOC);
        // Debug output
        //echo print_r($rows, true); // or var_dump($rows)
        $isDeleted = $rows['is_deleted'];
        $this->assertEquals('1', $isDeleted, 'Deleted should be equal to 1');

        $this->mockPdo->query("UPDATE meta_data SET is_deleted = 0 and is_active = 1 WHERE id = 1");
        $stmt = $this->mockPdo->query("SELECT is_deleted FROM meta_data      WHERE id = 1");
        $rows = $stmt->fetch(PDO::FETCH_ASSOC);
        $isDeleted = $rows['is_deleted'];
        $this->assertEquals('0', $isDeleted, 'Deleted should be equal to 1');
        
        $stmt = $this->mockPdo->prepare("SELECT * FROM meta_data_history WHERE meta_data_id = 1 and change_type = 'delete'");
        $stmt->execute();
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        //echo print_r($result, true); // or var_dump($result)
        $this->assertEquals('3', $result[0]['id']);
    }
    public function test018SetMetaDataToDeletedNoAccess(): void{
        $_SESSION['id'] = 1;                // User ID
        $_SESSION['roleValue'] = 6;         // Role Value (1 = SysAdmin)
        $_SESSION['roleTenantId'] = 1;      // Tenant ID for the role
        $_SESSION['tenantId'] = 1;          // Tenant ID for the user
        $_POST = [
            'action' => 'deleteMetaData',
            'data' => [
                'id' => 1,
                'isDeleted' => 1
            ],
            'debug' => 'CSD'
        ];
        list($output, $decodedOutput) = $this->executeFunctionPhp();
        $decodedOutput = json_decode($output, true);
        // Debug output
        //echo print_r($decodedOutput, true); // or var_dump($decodedOutput)
        $this->assertEquals('failed', $decodedOutput['status'], "Update MetaData as Id is missing: " . $output);
        $this->assertEquals('Access denied: insufficient permissions', $decodedOutput['message'], "ID 1 should be deactivated");
    }
/*
    public function testGetCurrentAccountInfo(){
        $stmt = $this->mockPdo->query("SELECT * FROM tenant_history");
        $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
        // Debug output
        print_r($rows); // or var_dump($rows)
        $this->assertEquals("1", "1");
    }
*/

    /* To Do: 
        - test view as sysadmin with a deleted tenant
        - test view as owner with a deleted tenant
        - test view as public with a deleted tenant
    */
}

?>