<?php

require '../vendor/autoload.php'; // Include Composer's autoloader

use HubSpot\Factory;
use HubSpot\Client\Crm\Contacts\ApiException;

function createAndAssociateNote($hubSpot, $contactId, $message) {
    try {
        // Ensure the message content is properly sanitized
        $message = htmlspecialchars(strip_tags($message));

        // Step 1: Create a note with required properties, including `hs_timestamp`
        $noteInput = [
            'properties' => [
                'hs_note_body' => $message, // Standard property for note content
                'hs_timestamp' => round(microtime(true) * 1000), // Current timestamp in milliseconds
            ],
        ];

        // Create the note
        $note = $hubSpot->crm()->objects()->basicApi()->create('notes', $noteInput);

        // Step 2: Associate the note with the contact using batch API
        $associations = [
            [
                'from' => ['id' => $note->getId()],
                'to' => ['id' => $contactId],
                'type' => 'note_to_contact', // Default association type
            ],
        ];

        $hubSpot->crm()->associations()->batchApi()->create(
            'notes', // fromObjectType
            'contacts', // toObjectType
            ['inputs' => $associations] // BatchInputPublicAssociation
        );

        return "Note successfully created and associated!";
    } catch (HubSpot\Client\Crm\Objects\ApiException $e) {
        // Log the full error response from the API
        return "Error: Unable to create or associate the note.";
    }
}



function hubSpotSend($inputData) {
    $accessToken = $inputData['accessToken']; // Dynamic access token

    // Initialize the HubSpot API client
    $hubSpot = Factory::createWithAccessToken($accessToken);

    // Form data
    $email = filter_var($inputData['email'], FILTER_VALIDATE_EMAIL);
    $phone = htmlspecialchars(strip_tags($inputData['phone'] ?? ''));
    $jobTitle = htmlspecialchars(strip_tags($inputData['jobTitle'] ?? ''));
    $message = htmlspecialchars(strip_tags($inputData['message'] ?? ''));
    $firstName = htmlspecialchars(strip_tags($inputData['firstName'] ?? ''));
    $lastName = htmlspecialchars(strip_tags($inputData['lastName'] ?? ''));

    if (!$email || !$message) {
        return "Email and message are required.";
    }

    try {
        // Step 1: Search for an existing contact by email
        $existingContacts = $hubSpot->crm()->contacts()->searchApi()->doSearch([
            'filterGroups' => [
                [
                    'filters' => [
                        [
                            'propertyName' => 'email',
                            'operator' => 'EQ',
                            'value' => $email,
                        ],
                    ],
                ],
            ],
        ]);

        if ($existingContacts->getTotal() > 0) {
            // Contact exists
            $existingContact = $existingContacts->getResults()[0];
            $contactId = $existingContact->getId();

            // Create and associate a note
            $data = createAndAssociateNote($hubSpot, $contactId, $message);
        } else {
            // Contact does not exist - create a new contact
            $contactInput = [
                'properties' => [
                    'email' => $email,
                    'phone' => $phone,
                    'firstname' => $firstName,
                    'lastname' => $lastName,
                    'jobtitle' => $jobTitle,
                ],
            ];

            $contact = $hubSpot->crm()->contacts()->basicApi()->create($contactInput);
            $contactId = $contact->getId();

            // Create and associate a note
            $data = createAndAssociateNote($hubSpot, $contactId, $message);
        }
    } catch (ApiException $e) {
        // Log detailed error information
        $data = "Error processing request: " . $e->getMessage();
    }

    return $data;
}
?>
