CREATE TABLE tenant (
    id INT AUTO_INCREMENT PRIMARY KEY,
    parent_id INT DEFAULT NULL,
    company_name VARCHAR(255) NOT NULL,
    domain VARCHAR(255) NOT NULL,
    company_number VARCHAR(255) NOT NULL,
    food_hygiene_id VARCHAR(255) NULL,
    primary_contact_name VARCHAR(255) NOT NULL,
    tenant_status ENUM('active', 'pending', 'suspended', 'deleted', 'hidden') DEFAULT 'pending',
    is_deleted TINYINT(1) DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;


CREATE TABLE tenant_contact (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tenant_id INT NOT NULL,
    user_id INT DEFAULT NULL,  -- Optional link to user (i.e. owner)
    contact_type ENUM('mobile', 'email', 'address', 'business') NOT NULL,
    is_primary ENUM('yes', 'no') NOT NULL DEFAULT 'no',
    label VARCHAR(255) NOT NULL,
    contact_value VARCHAR(255) NOT NULL,
    is_deleted TINYINT(1) DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (tenant_id) REFERENCES tenant(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES user_login(pk_user_id) ON DELETE SET NULL
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;


CREATE TABLE tenant_theme (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tenant_id INT NOT NULL,
    theme_name VARCHAR(100) NOT NULL,
    primary_color VARCHAR(7) NOT NULL,
    secondary_color VARCHAR(7) NOT NULL,
    text_color VARCHAR(7) NOT NULL,
    secondary_text_color VARCHAR(7) NOT NULL,
    accent_color VARCHAR(7) NOT NULL,
    background_color VARCHAR(7) NOT NULL,
    button_color VARCHAR(7) NOT NULL,
    theme_status ENUM('active', 'draft', 'archived') DEFAULT 'draft',
    is_applied ENUM('yes', 'no') DEFAULT 'no',
    is_deleted TINYINT(1) DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY uq_theme_name_per_tenant (tenant_id, theme_name),
    FOREIGN KEY (tenant_id) REFERENCES tenant(id) ON DELETE CASCADE
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

CREATE TABLE tenant_business_hours (
    id INT AUTO_INCREMENT PRIMARY KEY,
    tenant_id INT NOT NULL,
    hours_category ENUM('store', 'delivery', 'pickup', 'special') NOT NULL,   -- replaces `type`
    opening_day ENUM('Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday') DEFAULT NULL, -- replaces `day`
    specific_date DATE DEFAULT NULL,                    -- replaces `date` (used for special hours)
    open_time TIME DEFAULT NULL,                        -- replaces `start`
    close_time TIME DEFAULT NULL,                       -- replaces `end`
    is_closed TINYINT(1) DEFAULT 0,                     -- 1 = closed, 0 = open
    is_deleted TINYINT(1) DEFAULT 0,                    -- soft delete flag
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (tenant_id) REFERENCES tenant(id) ON DELETE CASCADE
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

CREATE TABLE tenant_history (
    id INT PRIMARY KEY AUTO_INCREMENT,
    tenant_id INTEGER NOT NULL,
    action_type ENUM('updated', 'deleted') NOT NULL,
    table_of_source VARCHAR(50) NOT NULL,  -- tenant, tenant_contact, etc.
    old_value TEXT DEFAULT NULL,    -- JSON string of old values
    action_by INT NOT NULL,        -- user_id or system
    action_at DATETIME DEFAULT CURRENT_TIMESTAMP
)ENGINE=InnoDB  DEFAULT CHARSET=latin1 COLLATE=latin1_swedish_ci;

ALTER TABLE tenant_business_hours
  ADD CONSTRAINT chk_hours_date_or_day
  CHECK (
    (specific_date IS NOT NULL AND opening_day IS NULL)
    OR (specific_date IS NULL AND opening_day IS NOT NULL)
  );

ALTER TABLE tenant_history
  ADD CONSTRAINT fk_action_by_user FOREIGN KEY (action_by) REFERENCES user_login(pk_user_id);


CREATE OR REPLACE VIEW tenant_upcoming_hours AS
SELECT
    d.generated_date,
    bh.tenant_id,
    bh.hours_category,
    CASE WHEN bh.is_closed = 1 THEN NULL ELSE bh.open_time END AS open_time,
    CASE WHEN bh.is_closed = 1 THEN NULL ELSE bh.close_time END AS close_time,
    bh.is_closed
FROM (
    -- Generate the next 14 days
    SELECT CURDATE() + INTERVAL seq DAY AS generated_date,
           DAYNAME(CURDATE() + INTERVAL seq DAY) AS day_name
    FROM (
        SELECT 0 AS seq UNION ALL SELECT 1 UNION ALL SELECT 2 UNION ALL SELECT 3 UNION ALL SELECT 4
        UNION ALL SELECT 5 UNION ALL SELECT 6 UNION ALL SELECT 7 UNION ALL SELECT 8 UNION ALL SELECT 9
        UNION ALL SELECT 10 UNION ALL SELECT 11 UNION ALL SELECT 12 UNION ALL SELECT 13
    ) AS days
) AS d
JOIN tenant_business_hours bh ON (
    (
        -- Special hours: take precedence
        bh.specific_date = d.generated_date
        AND bh.is_deleted = 0
    )
    OR (
        -- Regular hours: only used if NO special for that tenant on that date
        bh.specific_date IS NULL
        AND bh.opening_day = d.day_name
        AND bh.is_deleted = 0
        AND NOT EXISTS (
            SELECT 1 FROM tenant_business_hours sp
            WHERE 
                sp.tenant_id = bh.tenant_id
                AND sp.specific_date = d.generated_date
                AND sp.is_deleted = 0
        )
    )
)
ORDER BY bh.tenant_id, d.generated_date, bh.hours_category;


